package ru.yandex.direct.jobs.campaign;

import java.time.LocalDate;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.Nullable;

import one.util.streamex.EntryStream;
import org.jooq.Record;
import org.jooq.Result;
import org.jooq.impl.DSL;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.entity.client.model.Client;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsCpmPriceStatusApprove;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsCpmPriceStatusCorrect;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsType;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.jobs.campaign.service.ResetCpmPriceCampaignFlightStatusApproveMailSenderService;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.model.CheckTag;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.HourglassStretchPeriod;
import ru.yandex.direct.scheduler.support.DirectShardedJob;
import ru.yandex.direct.sender.YandexSenderException;

import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS;
import static ru.yandex.direct.dbschema.ppc.tables.CampaignsCpmPrice.CAMPAIGNS_CPM_PRICE;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_1;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapSet;
import static ru.yandex.direct.utils.StringUtils.nullIfBlank;

/**
 * Если пакетная кампания заапрувлена
 * И
 * если дата старта кампании - сегодня
 * И
 * если кампания не полная (не может быть запущена)
 * То
 * Сбрасываем статус брони в New
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(hours = 36),
        needCheck = ProductionOnly.class,
        tags = {DIRECT_PRIORITY_1, CheckTag.DIRECT_PRODUCT_TEAM}
)
// Запускается ежедневно в 4:00
@Hourglass(cronExpression = "0 0 4 * * ?", needSchedule = ProductionOnly.class)
@HourglassStretchPeriod(value = 600)
public class ResetCpmPriceCampaignFlightStatusApproveJob extends DirectShardedJob {

    private static final Logger logger = LoggerFactory.getLogger(ResetCpmPriceCampaignFlightStatusApproveJob.class);
    private static final Predicate<String> EMAIL_FILTER = email ->
            email.endsWith("@yandex-team.ru")
                    || email.endsWith("@yandex-team.com")
                    || email.endsWith("@yandex-team.com.ua")
                    || email.endsWith("@yandex-team.com.tr");

    private final DslContextProvider dslContextProvider;
    private final UserService userService;
    private final ResetCpmPriceCampaignFlightStatusApproveMailSenderService mailSenderService;
    private final ClientService clientService;

    @Autowired
    public ResetCpmPriceCampaignFlightStatusApproveJob(
            DslContextProvider dslContextProvider,
            UserService userService,
            ResetCpmPriceCampaignFlightStatusApproveMailSenderService mailSenderService,
            ClientService clientService) {
        this.dslContextProvider = dslContextProvider;
        this.userService = userService;
        this.mailSenderService = mailSenderService;
        this.clientService = clientService;
    }

    @Override
    public void execute() {
        Result<Record> campaignsToReset = getCpmPriceCampaignIdsToResetFlightStatusApprove(getShard());

        List<Long> campaignIds = campaignsToReset.getValues(CAMPAIGNS.CID);
        Map<Long, Long> cidToUid = campaignsToReset.intoMap(CAMPAIGNS.CID, CAMPAIGNS.UID);
        Map<Long, Long> cidToClientId = campaignsToReset.intoMap(CAMPAIGNS.CID, CAMPAIGNS.CLIENT_ID);
        // будет содержать 0, если кампания не ведётся агенством
        Map<Long, Long> cidToAgencyId = campaignsToReset.intoMap(CAMPAIGNS.CID, CAMPAIGNS.AGENCY_ID);
        Map<Long, LocalDate> cidToStartTime = campaignsToReset.intoMap(CAMPAIGNS.CID, CAMPAIGNS.START_TIME);

        resetFlightStatusApprove(getShard(),
                filterList(campaignIds, cid -> cidToStartTime.get(cid).isEqual(LocalDate.now())));
        declineNotShownCampaigns(getShard(),
                filterList(campaignIds, cid -> cidToStartTime.get(cid).isBefore(LocalDate.now())));

        Set<Long> allClientIds = Stream.of(cidToAgencyId.values(), cidToClientId.values())
                .flatMap(Collection::stream)
                .filter(clientId -> clientId != 0L)
                .collect(Collectors.toSet());
        Set<ClientId> allClientIds2 = mapSet(allClientIds, ClientId::fromLong);
        List<Client> allClients = clientService.massGetClient(allClientIds2);
        Map<Long, Client> clientIdToClient = listToMap(allClients, Client::getClientId);

        Map<Long, Long> cidToUidForEmail = new HashMap<>();
        Set<Long> alsoSendToDefaultEmailCids = new HashSet<>();
        campaignIds.forEach(cid -> {
            // В случае с агентским клиентом, надо отсылать менеджеру агентства, иначе менеджеру, иначе на общий e-mail
            // Если отсылаем менеджеру и этот менеджер под IDM, то дублируем письмо на общий e-mail
            Long agencyId = cidToAgencyId.get(cid);
            Long clientId = cidToClientId.get(cid);
            Client client = clientIdToClient.get(agencyId != 0 ? agencyId : clientId);
            Long primaryManagerUidForEmail = client.getPrimaryManagerUid();
            Boolean idm = client.getIsIdmPrimaryManager();
            cidToUidForEmail.put(cid, primaryManagerUidForEmail);
            if (Boolean.TRUE.equals(idm) && primaryManagerUidForEmail != null) {
                alsoSendToDefaultEmailCids.add(cid);
            }
        });

        sendEmails(cidToUidForEmail, alsoSendToDefaultEmailCids, cidToUid);
    }

    private void sendEmails(Map<Long, Long> cidToUidForEmail,
                            Set<Long> alsoSendToDefaultEmailCids,
                            Map<Long, Long> cidToUid) {
        Set<Long> allUids = Stream.of(cidToUidForEmail.values(), cidToUid.values())
                .flatMap(Collection::stream)
                .filter(Objects::nonNull)
                .collect(Collectors.toSet());
        Map<Long, User> uidToUser = listToMap(userService.massGetUser(allUids), User::getUid);

        Map<String, Set<Long>> emailToCids = buildEmailToCids(cidToUidForEmail, alsoSendToDefaultEmailCids, uidToUser);

        Map<Long, String> cidToLogin = EntryStream.of(cidToUid)
                .mapValues(uid -> uidToUser.get(uid).getLogin())
                .toMap();

        emailToCids.forEach((email, cids) -> {
            try {
                if (!mailSenderService.sendEmailResetFlightStatus(email, cids, cidToLogin)) {
                    logger.error("Couldn't send an email to address {}, bad email address", email);
                }
            } catch (YandexSenderException e) {
                logger.error("Couldn't send an email to client", e);
            }
        });

    }

    private static Map<String, Set<Long>> buildEmailToCids(Map<Long, Long> cidToUidForEmail,
                                                           Set<Long> alsoSendToDefaultEmailCids,
                                                           Map<Long, User> uidToUser) {
        Map<Long, String> cidToEmail = new HashMap<>();
        cidToUidForEmail.forEach((Long cid, @Nullable Long uid) ->
                cidToEmail.put(cid, nullIfBlank(
                        Optional.ofNullable(uidToUser.get(uid))
                                .map(User::getEmail)
                                .filter(EMAIL_FILTER)
                                .orElse(null))));

        Map<String, Set<Long>> emailToCids = new HashMap<>();
        cidToEmail.forEach((cid, email) ->
        {
            addCidToEmail(emailToCids, cid, email);
            if (alsoSendToDefaultEmailCids.contains(cid)) {
                addCidToEmail(emailToCids, cid, null);
            }
        });
        return emailToCids;
    }

    private static void addCidToEmail(Map<String, Set<Long>> emailToCids, Long cid, String email) {
        emailToCids.computeIfAbsent(email, key -> new HashSet<>())
                .add(cid);
    }

    private Result<Record> getCpmPriceCampaignIdsToResetFlightStatusApprove(int shard) {
        return dslContextProvider.ppc(shard)
                .select(List.of(CAMPAIGNS.CID, CAMPAIGNS.UID, CAMPAIGNS.CLIENT_ID, CAMPAIGNS.AGENCY_ID, CAMPAIGNS.START_TIME))
                .from(CAMPAIGNS)
                .join(CAMPAIGNS_CPM_PRICE).on(CAMPAIGNS_CPM_PRICE.CID.eq(CAMPAIGNS.CID))
                .where(CAMPAIGNS_CPM_PRICE.STATUS_APPROVE.eq(CampaignsCpmPriceStatusApprove.Yes)
                        .and(CAMPAIGNS_CPM_PRICE.STATUS_CORRECT.ne(CampaignsCpmPriceStatusCorrect.Yes))
                        .and(CAMPAIGNS.START_TIME.eq(DSL.currentLocalDate())
                                .or(CAMPAIGNS.START_TIME.lt(DSL.currentLocalDate())
                                        .and(DSL.nvl(CAMPAIGNS.SHOWS, 0L).eq(0L))))
                        .and(CAMPAIGNS.TYPE.eq(CampaignsType.cpm_price)))
                .fetch();
    }

    private void resetFlightStatusApprove(int shard, Collection<Long> campaignIds) {
        int rows = dslContextProvider.ppc(shard)
                .update(CAMPAIGNS_CPM_PRICE)
                .set(CAMPAIGNS_CPM_PRICE.STATUS_APPROVE, CampaignsCpmPriceStatusApprove.New)
                .where(CAMPAIGNS_CPM_PRICE.CID.in(campaignIds)
                        .and(CAMPAIGNS_CPM_PRICE.STATUS_CORRECT.ne(CampaignsCpmPriceStatusCorrect.Yes)))
                .execute();

        if (rows > 0) {
            logger.info("Reset flight status approve in shard {} cpm price campaigns: {}", shard, campaignIds);
        }
    }

    private void declineNotShownCampaigns(int shard, Collection<Long> campaignIds) {
        int rows = dslContextProvider.ppc(shard)
                .update(CAMPAIGNS_CPM_PRICE)
                .set(CAMPAIGNS_CPM_PRICE.STATUS_APPROVE, CampaignsCpmPriceStatusApprove.No)
                .where(CAMPAIGNS_CPM_PRICE.CID.in(campaignIds)
                        .and(CAMPAIGNS_CPM_PRICE.STATUS_CORRECT.ne(CampaignsCpmPriceStatusCorrect.Yes)))
                .execute();

        if (rows > 0) {
            logger.info("Declined flight status approve in shard {} cpm price campaigns: {}", shard, campaignIds);
        }
    }

}
