package ru.yandex.direct.jobs.campaign.cpmprice.unexpectedchanges;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod;
import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.core.entity.aggregatedstatuses.campaign.AggregatedStatusCampaignData;
import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.jobs.campaign.service.CpmPriceCampaignUnexpectedChangesMailService;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification;
import ru.yandex.direct.juggler.check.model.CheckTag;
import ru.yandex.direct.juggler.check.model.NotificationRecipient;
import ru.yandex.direct.rbac.RbacRole;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectJob;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YqlQuery;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.direct.ytwrapper.model.YtSQLSyntaxVersion;
import ru.yandex.direct.ytwrapper.model.YtTable;
import ru.yandex.direct.ytwrapper.utils.FreshestClusterFinder;
import ru.yandex.misc.io.ClassPathResourceInputStreamSource;

import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_2;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.utils.JsonUtils.fromJson;

@JugglerCheck(ttl = @JugglerCheck.Duration(days = 2),
        tags = {DIRECT_PRIORITY_2, CheckTag.YT},
        notifications = {
                @OnChangeNotification(recipient = NotificationRecipient.LOGIN_EBOGUSLAVSKAYA,
                        status = {JugglerStatus.OK, JugglerStatus.CRIT},
                        method = NotificationMethod.TELEGRAM),
        },
        needCheck = ProductionOnly.class
)
@Hourglass(cronExpression = "0 50 5 * * ?", needSchedule = ProductionOnly.class)
@ParametersAreNonnullByDefault
public class MonitoringCpmPriceUnexpectedChangesJob extends DirectJob {

    private static final String FIND_UNEXPECTED_CHANGED_CAMPAIGNS_QUERY = String.join("\n",
            new ClassPathResourceInputStreamSource(
                    "campaign/cpmprice/unexpected_changes/findUnexpectedChangedCampaigns.yql").readLines());

    private static final Logger logger = LoggerFactory.getLogger(MonitoringCpmPriceUnexpectedChangesJob.class);
    private final DirectConfig branchConfig;
    private final YtProvider ytProvider;
    private final CpmPriceCampaignUnexpectedChangesMailService mailSenderService;
    private final UserService userService;

    @Autowired
    public MonitoringCpmPriceUnexpectedChangesJob(
            DirectConfig directConfig,
            YtProvider ytProvider,
            CpmPriceCampaignUnexpectedChangesMailService mailSenderService, UserService userService) {
        this.branchConfig = directConfig.getBranch("cpm_price_campaign_mails");
        this.ytProvider = ytProvider;
        this.mailSenderService = mailSenderService;
        this.userService = userService;
    }

    @Override
    public void execute() {
        var ytBranch = branchConfig.getBranch("yt");

        var ytClusters = mapList(ytBranch.getStringList("clusters"),
                s -> YtCluster.valueOf(s.toUpperCase()));
        var ytMysqlSyncTables = new YtTable(ytBranch.getString("mysql_sync_states_table_path"));
        YtCluster ytCluster = FreshestClusterFinder.getFreshestCluster(ytProvider, ytMysqlSyncTables, ytClusters);
        if (ytCluster == null) {
            logger.error("Couldn't find available YT cluster, MonitoringCpmPriceUnexpectedChangesJob will be skipped");
            return;
        }
        var ytOperator = ytProvider.getOperator(ytCluster, YtSQLSyntaxVersion.SQLv1);

        var ytPool = ytBranch.getString("yt_pool");
        if (ytPool.isEmpty()) {
            ytPool = ytProvider.getClusterConfig(ytCluster).getUser();
        }
        var ytOutputTable = new YtTable(ytBranch.getString("unexpected_changes_table_partial_path"));
        var chunkSize = ytBranch.getInt("chunk_size");

        try (var ignore = Trace.current().profile("monitoring_campaign_changes:yql", "monitoring_changes")) {
            ytOperator.yqlExecute(
                    new YqlQuery(
                            FIND_UNEXPECTED_CHANGED_CAMPAIGNS_QUERY,
                            ytPool,
                            ytOutputTable.getPath())
                            .withTitle("MonitoringCpmPriceUnexpectedChangesJob:findUnexpectedChangedCampaigns"));
            logger.info("Temp data about campaigns saved in {}", ytOutputTable.getPath());
        }
        var tableRow = new MonitoringCpmPriceUnexpectedChangesTableRow();
        for (long chunkIndex = 0; ; chunkIndex++) {
            var monitoringCpmPriceUnexpectedChanges = new ArrayList<MonitoringCpmPriceUnexpectedChanges>();
            ytOperator.readTableByRowRange(
                    ytOutputTable,
                    row -> monitoringCpmPriceUnexpectedChanges.add(
                            new MonitoringCpmPriceUnexpectedChanges(
                                    row.getCid(),
                                    UnexpectedChangesReason.fromValue(row.getReason()),
                                    row.getManagerEmail(),
                                    row.getManagerLogin(),
                                    row.getUserLogin(),
                                    row.getClientName(),
                                    row.getCampaignName(),
                                    UnexpectedChangesProductType.fromValue(row.getProductType()),
                                    fromJson(row.getAggrData(), AggregatedStatusCampaignData.class)
                            )
                    ),
                    tableRow,
                    chunkSize * chunkIndex,
                    chunkSize * (chunkIndex + 1));

            var filteredMonitoringCpmPriceUnexpectedChanges = monitoringCpmPriceUnexpectedChanges.stream()
                    .filter(change ->
                    {
                        var user = userService.getUserByLogin(change.getUserLogin());
                        return user == null || !user.getRole().anyOf(RbacRole.SUPER);
                    })
                    .collect(Collectors.toCollection(ArrayList::new));

            if (filteredMonitoringCpmPriceUnexpectedChanges.isEmpty()) {
                break;
            }

            var emailToCampaigns = filteredMonitoringCpmPriceUnexpectedChanges
                    .stream()
                    .filter(m -> m.getManagerEmail() != null)
                    .collect(Collectors.groupingBy(MonitoringCpmPriceUnexpectedChanges::getManagerEmail,
                            Collectors.toList()));
            for (Map.Entry<String, List<MonitoringCpmPriceUnexpectedChanges>> field : emailToCampaigns.entrySet()) {
                mailSenderService.sendEmailToManager(field.getKey(), field.getValue());
            }

            mailSenderService.sendEmailToCommon(filteredMonitoringCpmPriceUnexpectedChanges);
        }

    }

}
