package ru.yandex.direct.jobs.campaign.paused.daybudget;

import java.util.ArrayList;
import java.util.EnumSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.collections4.ListUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification;
import ru.yandex.direct.juggler.check.model.NotificationRecipient;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectShardedJob;

/**
 * Джоба, отправляющая уведомления кампаниям, остановленным по дневному бюджету
 *
 * @see
 * <a href="https://a.yandex-team.ru/arc/trunk/arcadia/direct/perl/protected/DayBudgetAlerts.pm?rev=r7939428#L59-122">
 * Оригинал
 * </a>
 */
@JugglerCheck(
        ttl = @JugglerCheck.Duration(hours = 1),
        notifications = @OnChangeNotification(
                method = NotificationMethod.TELEGRAM,
                recipient = NotificationRecipient.CHAT_API_CLEAN_MONITORING,
                status = {JugglerStatus.OK, JugglerStatus.CRIT}
        ),
        needCheck = ProductionOnly.class
)
@Hourglass(periodInSeconds = 60, needSchedule = ProductionOnly.class)
@ParametersAreNonnullByDefault
public class PausedByDayBudgetCampaignsWarningsSenderJob extends DirectShardedJob {
    private static final int CHUNK_SIZE = 300;

    private static final Logger logger = LoggerFactory.getLogger(PausedByDayBudgetCampaignsWarningsSenderJob.class);

    private final CampaignRepository campaignRepository;

    private final PausedByDayBudgetService pausedByDayBudgetService;
    private final PausedByDayBudgetSenderService senderService;


    public PausedByDayBudgetCampaignsWarningsSenderJob(int shard,
                                                       PausedByDayBudgetService pausedByDayBudgetService,
                                                       CampaignRepository campaignRepository,
                                                       PausedByDayBudgetSenderService senderService) {
        super(shard);
        this.pausedByDayBudgetService = pausedByDayBudgetService;
        this.campaignRepository = campaignRepository;
        this.senderService = senderService;
    }

    @Autowired
    public PausedByDayBudgetCampaignsWarningsSenderJob(PausedByDayBudgetService pausedByDayBudgetService,
                                                       CampaignRepository campaignRepository,
                                                       PausedByDayBudgetSenderService senderService) {
        this.pausedByDayBudgetService = pausedByDayBudgetService;
        this.campaignRepository = campaignRepository;
        this.senderService = senderService;
    }

    private boolean addNotificationAndCheckSuccess(Campaign campaign,
                                                   Map<Long, Set<PausedByDayBudgetNotificationType>> notificationTypesById) {
        var set = notificationTypesById.getOrDefault(
                campaign.getId(),
                EnumSet.noneOf(PausedByDayBudgetNotificationType.class)
        );
        return senderService.addNotification(
                getShard(), campaign, false, PausedByDayBudgetNotificationType.retainSuitableForCampaignTypes(set));
    }


    @Override
    public void execute() {
        List<Campaign> allCampaigns =
                new ArrayList<>(campaignRepository.getCampaignsReadyForDayBudgetNotifications(getShard()));

        for (List<Campaign> campaigns : ListUtils.partition(allCampaigns, CHUNK_SIZE)) {
            logger.info("Got {} campaigns", campaigns.size());

            Map<Long, Set<PausedByDayBudgetNotificationType>> allowedNotificationTypes =
                    pausedByDayBudgetService.getAllowedNotificationTypesById(campaigns);

            var notifiedCampaignIds = pausedByDayBudgetService
                    .getSuitableCampaigns(getShard(), campaigns)
                    .filter(c -> addNotificationAndCheckSuccess(c, allowedNotificationTypes))
                    .map(Campaign::getId)
                    .collect(Collectors.toSet());

            logger.info("Notified {} campaigns", notifiedCampaignIds.size());
            campaignRepository.markDayBudgetNotificationsAsSent(getShard(), notifiedCampaignIds);
        }
    }
}
