package ru.yandex.direct.jobs.campaign.paused.daybudget;

import java.util.Map;
import java.util.Set;

import org.apache.commons.validator.routines.EmailValidator;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.TranslationService;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.model.SmsFlag;
import ru.yandex.direct.core.entity.eventlog.service.EventLogService;
import ru.yandex.direct.core.entity.notification.LocaleResolver;
import ru.yandex.direct.core.entity.notification.repository.SmsQueueRepository;
import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.sender.YandexSenderClient;
import ru.yandex.direct.sender.YandexSenderException;
import ru.yandex.direct.sender.YandexSenderTemplateParams;

/**
 * Сервис предоставляющий набор методов полезных для отправки различного рода уведомлений, используемых джобами
 * {@link PausedByDayBudgetCampaignsWarningsSenderJob} и {@link PausedByDayBudgetWalletsWarningsSenderJob}
 */
@Service
public class PausedByDayBudgetSenderService {

    public static final String CLIENT_ID_KEY = "ClientID";
    public static final String LOGIN_KEY = "val1";
    public static final String CAMPAIGN_ID_KEY = "val2";
    public static final String MENTION_CID_KEY = "val3";

    public static final String MENTION_CID_VALUE = "";
    public static final String NOT_MENTION_CID_VALUE = "true";

    private static final Logger logger = LoggerFactory.getLogger(PausedByDayBudgetSenderService.class);

    private final UserService userService;
    private final YandexSenderClient senderClient;
    private final PausedByDayBudgetCampaignsWarningsMailTemplateResolver mailTemplateResolver;
    private final EventLogService eventLogService;
    private final SmsQueueRepository smsQueueRepository;
    private final TranslationService translationService;
    private final EmailValidator emailValidator = EmailValidator.getInstance();

    public PausedByDayBudgetSenderService(UserService userService, YandexSenderClient senderClient,
                                          PausedByDayBudgetCampaignsWarningsMailTemplateResolver mailTemplateResolver,
                                          EventLogService eventLogService, SmsQueueRepository smsQueueRepository,
                                          TranslationService translationService) {
        this.userService = userService;
        this.senderClient = senderClient;
        this.mailTemplateResolver = mailTemplateResolver;
        this.eventLogService = eventLogService;
        this.smsQueueRepository = smsQueueRepository;
        this.translationService = translationService;
    }

    public enum MailSendingResult {
        //случилась ошибка, во время отправки, значит можно попозже попробовать еще раз
        TRY_AGAIN,

        //случилась непоправимая ошибка, пробовать еще раз бессмысленно
        ERROR,

        //все получилось
        SUCCESS
    }

    /**
     * Добавляет уведомление о том, что кампания или ОС остановлен по дневному лимиту
     *
     * @param shard                    шард
     * @param campaign                 остановленная кампания
     * @param isWalletCampaign         кампания - ОС или нет
     * @param allowedNotificationTypes множество доступных каналов связи с пользователем
     * @return получилось ли отправить уведомление
     *
     * @implNote Если при отправке email ошибка произошла на этап рассылки, то возвращаем {@code false}, чтобы
     * попробовать еще раз
     */
    public boolean addNotification(int shard,
                                   @NotNull Campaign campaign,
                                   boolean isWalletCampaign,
                                   @NotNull Set<PausedByDayBudgetNotificationType> allowedNotificationTypes) {
        // Для кошельков не надо в письме указывать никакой cid, для некошельков нужно
        boolean sent = false;
        MailSendingResult result = allowedNotificationTypes.contains(PausedByDayBudgetNotificationType.EMAIL) ?
                sendMail(campaign, !isWalletCampaign) : null;

        if (MailSendingResult.SUCCESS.equals(result)) {
            sent = true;
        }

        if (MailSendingResult.TRY_AGAIN.equals(result)) {
            return false;
        }

        // Если result Error или null, то письмом не получится отправить уведомление,
        // пробуем другим способом
        if (allowedNotificationTypes.contains(PausedByDayBudgetNotificationType.SMS)) {
            sent |= sendSms(shard, campaign);
        }
        if (allowedNotificationTypes.contains(PausedByDayBudgetNotificationType.EVENT_LOG)) {
            sent |= addToEventLog(campaign);
        }
        return sent;
    }


    /**
     * Делает запись в {@link ru.yandex.direct.core.entity.eventlog.model.EventLog}
     * об остановленной по дневному бюджету кампании
     *
     * @param campaign кампания, остановленная по дневному бюджету
     */
    public boolean addToEventLog(@NotNull Campaign campaign) {
        eventLogService.addPausedByDayBudgetEventLog(
                campaign.getType(),
                campaign.getClientId(),
                campaign.getId(),
                campaign.getDayBudgetStopTime()
        );
        return true;
    }

    /**
     * Отправляет уведомление об остановке по дневному бюджету по email
     *
     * @param campaign       кампания, из которой берутся данные о том, куда и какое письмо отправлять.
     * @param needMentionCid флаг, говорящий о том, кампания - общий счет или нет. Из чего однозначно следует нужно
     *                       ли в письме указывать id кампании
     * @return {@code MailSendingResult.ERROR}, если некорректны данные отправки. <br>
     *         {@code MailSendingResult.TRY_AGAIN}, если что-то сломалось у {@code senderClient}. <br>
     *         {@code MailSendingResult.SUCCESS}, если удалось отправить письмо
     */
    public MailSendingResult sendMail(@NotNull Campaign campaign, boolean needMentionCid) {
        var user = userService.getUser(campaign.getUserId());
        if (user == null) {
            logger.error("user null for campaign {}", campaign.getId());
            return MailSendingResult.ERROR;
        }
        String email = user.getEmail();
        if (!emailValidator.isValid(email)) {
            logger.error("incorrect email for campaign {}", campaign.getId());
            return MailSendingResult.ERROR;
        }
        String campSlug = mailTemplateResolver.getTemplateByLanguage(user.getLang());
        if (campSlug == null) {
            logger.error("invalid campSlug for campaign {}", campaign.getId());
            return MailSendingResult.ERROR;
        }
        YandexSenderTemplateParams senderTemplateParams = new YandexSenderTemplateParams.Builder()
                .withCampaignSlug(campSlug)
                .withAsync(Boolean.TRUE)
                .withToEmail(email)
                .withArgs(Map.ofEntries(
                        Map.entry(CLIENT_ID_KEY, user.getClientId().toString()),
                        Map.entry(LOGIN_KEY, user.getLogin()),
                        Map.entry(CAMPAIGN_ID_KEY, campaign.getId().toString()),
                        Map.entry(MENTION_CID_KEY, needMentionCid ?
                                MENTION_CID_VALUE :
                                NOT_MENTION_CID_VALUE)))
                .build();
        try {
            return senderClient.sendTemplate(senderTemplateParams, YandexSenderClient::isInvalidToEmail) ?
                    MailSendingResult.SUCCESS : MailSendingResult.TRY_AGAIN;
        } catch (YandexSenderException e) {
            logger.error("failed sending main", e);
            return MailSendingResult.TRY_AGAIN;
        }
    }

    /**
     * Отправляет уведомление об остановке по дневному бюджету по SMS.
     *
     * @param shard    шард
     * @param campaign кампания остановленная по дневному бюджету
     */
    public boolean sendSms(int shard, @NotNull Campaign campaign) {
        var user = userService.getUser(campaign.getUserId());
        if (user == null) {
            logger.error("user is null for campaign {}", campaign.getId());
            return false;
        }
        String message = translationService.translate(
                PausedByDayBudgetSmsTranslations.INSTANCE.pausedByDayBudgetWalletWarning(),
                LocaleResolver.getLocaleByLanguageWithFallback(user.getLang())
        );

        smsQueueRepository.addToSmsQueue(
                shard, user.getUid(), campaign.getId(),
                message, SmsFlag.PAUSED_BY_DAY_BUDGET_SMS, campaign.getSmsTime()
        );
        return true;
    }
}
