package ru.yandex.direct.jobs.campaign.paused.daybudget;

import java.math.BigDecimal;
import java.time.Instant;
import java.time.ZoneId;
import java.util.Collection;
import java.util.EnumSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.jetbrains.annotations.NotNull;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.model.CampaignEmailNotification;
import ru.yandex.direct.core.entity.campaign.model.SmsFlag;
import ru.yandex.direct.core.entity.campaign.model.TimeTargetStatus;
import ru.yandex.direct.core.entity.campaign.model.WalletCampaign;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.campaign.service.TimeTargetStatusService;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.timetarget.model.GeoTimezone;
import ru.yandex.direct.core.entity.timetarget.repository.GeoTimezoneRepository;
import ru.yandex.direct.currency.Currencies;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.regions.Region;

import static ru.yandex.direct.utils.FunctionalUtils.filterAndMapToSet;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;

/**
 * Сервис предоставляющий набор методов полезных для обработки кампаний, используемый джобами
 * {@link PausedByDayBudgetCampaignsWarningsSenderJob} и {@link PausedByDayBudgetWalletsWarningsSenderJob}
 */
@Service
public class PausedByDayBudgetService {
    private final TimeTargetStatusService timeTargetStatusService;
    private final GeoTimezoneRepository geoTimezoneRepository;
    private final FeatureService featureService;
    private final CampaignRepository campaignRepository;

    @Autowired
    public PausedByDayBudgetService(TimeTargetStatusService timeTargetStatusService,
                                    GeoTimezoneRepository geoTimezoneRepository, FeatureService featureService,
                                    CampaignRepository campaignRepository) {
        this.timeTargetStatusService = timeTargetStatusService;
        this.geoTimezoneRepository = geoTimezoneRepository;
        this.featureService = featureService;
        this.campaignRepository = campaignRepository;
    }

    /**
     * Возвращает мапу, отображающую айдишники кампаний в множество способов
     * {@link PausedByDayBudgetNotificationType}, которыми можно отправить уведомление.
     *
     * @param campaigns кампании для которых нужно получить способы
     * @return отображение айдишников в возможные способы нотификации
     */
    public Map<Long, Set<PausedByDayBudgetNotificationType>> getAllowedNotificationTypesById(List<Campaign> campaigns) {

        Set<ClientId> clients = listToSet(campaigns, c -> ClientId.fromLong(c.getClientId()));
        Map<ClientId, Boolean> featureEnabled = featureService.isEnabledForClientIdsOnlyFromDb(clients,
                FeatureName.PAUSED_BY_DAY_BUDGET_WARNINGS.getName());
        return campaigns.stream()
                .collect(Collectors.toMap(
                        Campaign::getId,
                        c -> {
                            var clientId = ClientId.fromLong(c.getClientId());
                            if (Boolean.FALSE.equals(featureEnabled.get(clientId))) {
                                return EnumSet.noneOf(PausedByDayBudgetNotificationType.class);
                            }

                            var set = EnumSet.of(PausedByDayBudgetNotificationType.EVENT_LOG);

                            if (c.getEmailNotifications().contains(CampaignEmailNotification.PAUSED_BY_DAY_BUDGET)) {
                                set.add(PausedByDayBudgetNotificationType.EMAIL);
                            }

                            if (c.getSmsFlags().contains(SmsFlag.PAUSED_BY_DAY_BUDGET_SMS)) {
                                set.add(PausedByDayBudgetNotificationType.SMS);
                            }

                            return set;
                        })
                );
    }

    /**
     * Из полученных кампаний убирает те, которые не подходят под условие нотификации.
     *
     * <p>Используемые условия для фильтрации:
     * {@link PausedByDayBudgetService#haveEnoughMoneyRest(Campaign, Map, Map)},
     * {@link PausedByDayBudgetService#isActiveCampaign(Campaign)}</p>
     *
     * @param shard     шард
     * @param campaigns кампании, которые нужно отфильтровать
     * @return поток (для удобства) кампаний подходящих под условия нотификации
     */
    public Stream<Campaign> getSuitableCampaigns(int shard, Collection<Campaign> campaigns) {
        Set<Long> walletCampaignIds = filterAndMapToSet(
                campaigns, c -> c.getWalletId() > 0, Campaign::getWalletId);

        Map<Long, BigDecimal> sumByWalletId = campaignRepository
                .getWalletsByWalletCampaignIds(shard, walletCampaignIds)
                .stream()
                .collect(Collectors.toMap(WalletCampaign::getId, WalletCampaign::getSum));
        Map<Long, BigDecimal> debtsByWalletIds = campaignRepository.getWalletsDebt(shard, walletCampaignIds);

        return campaigns.stream()
                .filter(this::isActiveCampaign)
                .filter(c -> haveEnoughMoneyRest(c, sumByWalletId, debtsByWalletIds));
    }


    /**
     * Проверяет для кампании, может ли она крутиться по наличию средств.
     * Иными словами проверят есть ли у кампании деньги доступные к трате
     *
     * @param campaign        кампания
     * @param sumByWalletId   сумма по айди кошелька
     * @param debtsByWalletId долги по айди кошелька
     * @return есть ли у кампании деньги доступные к трате
     * @see
     * <a href="https://a.yandex-team.ru/arc/trunk/arcadia/direct/perl/protected/DayBudgetAlerts.pm?rev=r7939428#L232-275">
     * Оригинал
     * </a>
     */
    private boolean haveEnoughMoneyRest(@NotNull Campaign campaign, @NotNull Map<Long, BigDecimal> sumByWalletId,
                                        @NotNull Map<Long, BigDecimal> debtsByWalletId) {
        var rest = sumByWalletId.getOrDefault(campaign.getWalletId(), BigDecimal.ZERO)
                .add(debtsByWalletId.getOrDefault(campaign.getWalletId(), BigDecimal.ZERO));

        var sumTotalOrZero = campaign.getSum().subtract(campaign.getSumSpent()).max(BigDecimal.ZERO);

        return rest.add(sumTotalOrZero).compareTo(Currencies.EPSILON) > 0;
    }

    /**
     * Проверяет не остановлена ли кампания по временному таргетингу
     *
     * @param campaign кампания
     * @return не остановлена ли кампания
     * @see
     * <a href="https://a.yandex-team.ru/arc/trunk/arcadia/direct/perl/protected/DayBudgetAlerts.pm?rev=r7939428#L214-230">
     * Оригинал
     * </a>
     */
    private boolean isActiveCampaign(@NotNull Campaign campaign) {
        var geoTimezone = geoTimezoneRepository
                .getGeoTimezonesByTimezoneIds(List.of(campaign.getTimezoneId()))
                .stream()
                .findAny().orElse(new GeoTimezone()
                        .withTimezone(ZoneId.of("Europe/Moscow"))
                        .withRegionId(Region.RUSSIA_REGION_ID)
                );

        var status = timeTargetStatusService.getTimeTargetStatus(
                campaign.getTimeTarget(), geoTimezone, Instant.now()).getStatus();

        return status == TimeTargetStatus.ACTIVE;
    }

}
