package ru.yandex.direct.jobs.campaign.paused.daybudget;

import java.util.ArrayList;
import java.util.EnumSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.collections4.ListUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification;
import ru.yandex.direct.juggler.check.model.NotificationRecipient;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectShardedJob;

/**
 * Джоба, отправляющая уведомления кошелькам остановленным по дневному бюджету
 *
 * @see
 * <a href="https://a.yandex-team.ru/arc/trunk/arcadia/direct/perl/protected/DayBudgetAlerts.pm?rev=r7939428#L124-212">
 * Оригинал
 * </a>
 */
@JugglerCheck(
        ttl = @JugglerCheck.Duration(hours = 1),
        notifications = @OnChangeNotification(
                method = NotificationMethod.TELEGRAM,
                recipient = NotificationRecipient.CHAT_API_CLEAN_MONITORING,
                status = {JugglerStatus.OK, JugglerStatus.CRIT}
        ),
        needCheck = ProductionOnly.class
)
@Hourglass(periodInSeconds = 60, needSchedule = ProductionOnly.class)
@ParametersAreNonnullByDefault
public class PausedByDayBudgetWalletsWarningsSenderJob extends DirectShardedJob {
    private static final int CHUNK_SIZE = 300;

    private static final Logger logger = LoggerFactory.getLogger(PausedByDayBudgetWalletsWarningsSenderJob.class);

    private final CampaignRepository campaignRepository;

    private final PausedByDayBudgetService pausedByDayBudgetService;
    private final PausedByDayBudgetSenderService senderService;

    public PausedByDayBudgetWalletsWarningsSenderJob(int shard,
                                                     PausedByDayBudgetService pausedByDayBudgetService,
                                                     PausedByDayBudgetSenderService senderService,
                                                     CampaignRepository campaignRepository) {
        super(shard);
        this.pausedByDayBudgetService = pausedByDayBudgetService;
        this.senderService = senderService;
        this.campaignRepository = campaignRepository;
    }

    @Autowired
    public PausedByDayBudgetWalletsWarningsSenderJob(PausedByDayBudgetService pausedByDayBudgetService,
                                                     PausedByDayBudgetSenderService senderService,
                                                     CampaignRepository campaignRepository) {
        this.pausedByDayBudgetService = pausedByDayBudgetService;
        this.senderService = senderService;
        this.campaignRepository = campaignRepository;
    }

    public boolean addNotificationAndCheckSuccess(Campaign wallet,
                                                  Map<Long, Set<PausedByDayBudgetNotificationType>> notificationTypesByWalletId) {
        var set = notificationTypesByWalletId.getOrDefault(
                wallet.getId(),
                EnumSet.noneOf(PausedByDayBudgetNotificationType.class)
        );

        return senderService.addNotification(
                getShard(), wallet, true, PausedByDayBudgetNotificationType.retainSuitableForWalletTypes(set));
    }

    @Override
    public void execute() {
        List<Campaign> allCampaignWallets =
                new ArrayList<>(campaignRepository.getWalletsReadyForDayBudgetNotifications(getShard()));

        for (List<Campaign> wallets : ListUtils.partition(allCampaignWallets, CHUNK_SIZE)) {
            logger.info("Got {} wallets", wallets.size());

            Map<Long, Set<PausedByDayBudgetNotificationType>> allowedNotificationTypes =
                    pausedByDayBudgetService.getAllowedNotificationTypesById(wallets);

            Map<Long, List<Campaign>> campaignsByWalletId = campaignRepository
                    .getCampaignsUnderWalletsReadyForDayBudgetNotification(getShard(), wallets)
                    .stream()
                    .collect(Collectors.groupingBy(Campaign::getWalletId, Collectors.toList()));

            List<Long> notifiedCampaignIds = wallets.stream()
                    .filter(wc -> pausedByDayBudgetService.getSuitableCampaigns(
                                    getShard(), campaignsByWalletId.getOrDefault(wc.getId(), List.of()))
                            .findAny().isPresent())
                    .filter(wc -> addNotificationAndCheckSuccess(wc, allowedNotificationTypes))
                    .map(Campaign::getId)
                    .collect(Collectors.toList());

            logger.info("Notified {} wallets", notifiedCampaignIds.size());
            campaignRepository.markDayBudgetNotificationsAsSent(getShard(), notifiedCampaignIds);
        }
    }
}
