package ru.yandex.direct.jobs.campaign.service;

import java.util.List;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.jobs.campaign.cpmprice.unexpectedchanges.MonitoringCpmPriceUnexpectedChanges;
import ru.yandex.direct.sender.YandexSenderClient;

import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
@ParametersAreNonnullByDefault
public class CpmPriceCampaignUnexpectedChangesMailService extends CpmPriceCampaignMailSenderService{
    private static final Logger logger = LoggerFactory.getLogger(
            CpmPriceCampaignUnexpectedChangesMailService.class);

    @Autowired
    public CpmPriceCampaignUnexpectedChangesMailService(
            DirectConfig config,
            YandexSenderClient senderClient) {
        super(config, senderClient);
    }
    public boolean sendEmailToManager(String email, List<MonitoringCpmPriceUnexpectedChanges> changes) {
        if (changes.isEmpty()) {
            return true;
        }
        logger.info("Sending email to manager {} that some campaigns have unexpected status changes {}", email, changes.get(0).getCampaignId());
        var campaignsHtmlTable = getCampaignsHtmlTable(changes);
        var argsManager = Map.of(
                "cpm_price_campaigns_list_unexpected_changed", campaignsHtmlTable,
                "ClientID", "",
                "debug_info", getCommonDebugInfo(email));

        var slugManager = branch.getString("UNEXPECTED_CHANGES_MANAGER");
        return sendEmail(email, slugManager, argsManager);
    }

    public boolean sendEmailToCommon(List<MonitoringCpmPriceUnexpectedChanges> changes) {
        if (changes.isEmpty()) {
            return true;
        }
        logger.info("Sending common letter that some campaigns have unexpected status changes: {}",
                mapList(changes, MonitoringCpmPriceUnexpectedChanges::getCampaignId));
        var campaignsHtmlTable = getCampaignsHtmlTable(changes);

        var argsCommon = Map.of(
                "cpm_price_campaigns_list_unexpected_changed", campaignsHtmlTable,
                "ClientID", "",
                "debug_info", getCommonDebugInfo(defaultEmail));

        var slugCommon = branch.getString("UNEXPECTED_CHANGES_COMMON");
        return sendEmail(defaultEmail, slugCommon, argsCommon);
    }

    private String getCampaignsHtmlTable(List<MonitoringCpmPriceUnexpectedChanges> changes) {
        StringBuilder html = new StringBuilder();

        for (var change : changes) {
            html.append("<tr><td>")
                    .append(getUrlToViewCampaign(change.getCampaignId(), change.getUserLogin()))
                    .append("</td><td>")
                    .append(nvl(change.getManagerLogin(), "--"))
                    .append("</td><td>")
                    .append(getReason(change))
                    .append("</td><td>")
                    .append(change.getProductType().getText())
                    .append("</td><td>")
                    .append(change.getCampaignName())
                    .append("</td><td>")
                    .append(nvl(change.getClientName(), "--"))
                    .append("</td></tr>");
        }

        return html.toString();
    }

}
