package ru.yandex.direct.jobs.campautoprice;

import java.time.LocalDateTime;
import java.util.List;

import com.google.common.collect.Iterables;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.common.util.RelaxedWorker;
import ru.yandex.direct.core.entity.autoprice.repository.AutoPriceCampQueueRepository;
import ru.yandex.direct.env.TypicalEnvironment;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectShardedJob;

import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_2;
import static ru.yandex.direct.juggler.check.model.CheckTag.GROUP_INTERNAL_SYSTEMS;
import static ru.yandex.direct.juggler.check.model.CheckTag.JOBS_RELEASE_REGRESSION;

/**
 * <p>Очищает таблицу {@link ru.yandex.direct.dbschema.ppc.tables.AutoPriceCampQueue}</p> от заданий,
 * которые не ожидают обработки и были добавлены раньше, чем {@code BORDER_DAYS} дней назад
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(days = 2, hours = 4),
        tags = {DIRECT_PRIORITY_2, GROUP_INTERNAL_SYSTEMS, JOBS_RELEASE_REGRESSION})
@Hourglass(cronExpression = "0 0 3 * * ?", needSchedule = TypicalEnvironment.class)
public class CampAutoPriceQueueCleaner extends DirectShardedJob {
    private static final int DELETE_CHUNK_SIZE = 10_000;
    private static final RelaxedWorker relaxedWorker = new RelaxedWorker(3.0);
    private static final int BORDER_DAYS = 7;

    private static final Logger logger = LoggerFactory.getLogger(CampAutoPriceQueueCleaner.class);

    private final AutoPriceCampQueueRepository autoPriceCampQueueRepository;

    @Autowired
    public CampAutoPriceQueueCleaner(AutoPriceCampQueueRepository autoPriceCampQueueRepository) {
        this.autoPriceCampQueueRepository = autoPriceCampQueueRepository;
    }

    @Override
    public void execute() {
        LocalDateTime borderDateTime = LocalDateTime.now().minusDays(BORDER_DAYS);

        logger.debug("get outdated queue records");
        List<Long> outdatedIds = autoPriceCampQueueRepository.getOutdatedIds(getShard(), borderDateTime);
        for (List<Long> chunk : Iterables.partition(outdatedIds, DELETE_CHUNK_SIZE)) {
            logger.debug("delete next records chunk");
            logger.trace("Ids for delete: {}", chunk);
            relaxedWorker.runAndRelax(() -> {
                int deletedRows = autoPriceCampQueueRepository.deleteByIds(getShard(), chunk);
                logger.info("deleted {} records", deletedRows);
            });
        }
    }
}
