package ru.yandex.direct.jobs.campdaybudgethistory;

import java.time.Duration;
import java.time.LocalDateTime;
import java.util.List;

import com.google.common.collect.Iterables;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.common.util.RelaxedWorker;
import ru.yandex.direct.core.entity.campdaybudgethistory.repository.CampDayBudgetStopHistoryRepository;
import ru.yandex.direct.env.TypicalEnvironment;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.model.CheckTag;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectShardedJob;
import ru.yandex.direct.utils.TimeProvider;

import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_2;

/**
 * Ежедневное удаление старой истории остановок кампаний по дневному бюджету.
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(days = 2, hours = 4), tags = {DIRECT_PRIORITY_2, CheckTag.GROUP_INTERNAL_SYSTEMS})
@Hourglass(cronExpression = "0 52 4 * * ?", needSchedule = TypicalEnvironment.class)
class ClearCampDayBudgetStopHistoryJob extends DirectShardedJob {

    private static final Logger logger = LoggerFactory.getLogger(ClearCampDayBudgetStopHistoryJob.class);
    private static final RelaxedWorker RELAXED_WORKER = new RelaxedWorker(2.0);
    /**
     * за какой период храним историю.
     */
    private static final Duration HISTORY_EXPIRE_DURATION = Duration.ofDays(30);

    /**
     * Максимальное количество записей, выбираемых за один запрос.
     */
    private final int selectLimit;
    /**
     * Максимальное количество записей, удаляемых за один запросом.
     */
    private final int deleteLimit;

    private final CampDayBudgetStopHistoryRepository campDayBudgetStopHistoryRepository;
    private final TimeProvider timeProvider;

    @Autowired
    ClearCampDayBudgetStopHistoryJob(CampDayBudgetStopHistoryRepository campDayBudgetStopHistoryRepository) {
        this.campDayBudgetStopHistoryRepository = campDayBudgetStopHistoryRepository;
        this.timeProvider = new TimeProvider();
        selectLimit = 10_000;
        deleteLimit = 500;
    }

    /**
     * Конструктор нужен для тестов для указания шарда и timeProvider.
     */
    ClearCampDayBudgetStopHistoryJob(int shard, CampDayBudgetStopHistoryRepository campDayBudgetStopHistoryRepository,
            TimeProvider timeProvider, int selectLimit, int deleteLimit)
    {
        super(shard);
        this.campDayBudgetStopHistoryRepository = campDayBudgetStopHistoryRepository;
        this.timeProvider = timeProvider;
        this.selectLimit = selectLimit;
        this.deleteLimit = deleteLimit;
    }

    @Override
    public void execute() {
        LocalDateTime historyExpireDateTime = timeProvider.now().minus(HISTORY_EXPIRE_DURATION);
        List<Long> ids;
        int totalDeleted = 0;
        do {
            ids = campDayBudgetStopHistoryRepository.getIdsOfExpiredRecords(getShard(),
                    historyExpireDateTime, selectLimit);
            logger.info("Selected {} records to delete", ids.size());
            for (List<Long> chunk : Iterables.partition(ids, deleteLimit)) {
                totalDeleted += RELAXED_WORKER.callAndRelax(() ->
                        campDayBudgetStopHistoryRepository.delete(getShard(), chunk));
            }
        } while (ids.size() == selectLimit);
        logger.info("Totally deleted {} records", totalDeleted);
    }
}
