package ru.yandex.direct.jobs.campdaybudgetlimitstoptime

import org.jooq.Configuration
import org.jooq.Query
import org.jooq.impl.DSL
import org.slf4j.LoggerFactory
import org.springframework.beans.factory.config.ConfigurableBeanFactory.SCOPE_PROTOTYPE
import org.springframework.context.annotation.Scope
import org.springframework.stereotype.Component
import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod
import ru.yandex.direct.common.db.PpcPropertiesSupport
import ru.yandex.direct.common.db.PpcPropertyName
import ru.yandex.direct.common.db.PpcPropertyNames.CAMP_DAY_BUDGET_LIMIT_STOP_TIME_JAVA_WRITING_ENABLED
import ru.yandex.direct.common.db.PpcPropertyType
import ru.yandex.direct.config.DirectConfig
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository
import ru.yandex.direct.dbschema.ppc.Tables.CAMP_DAY_BUDGET_STOP_HISTORY
import ru.yandex.direct.dbschema.ppc.Tables.CAMP_OPTIONS
import ru.yandex.direct.dbschema.ppc.enums.CampOptionsDayBudgetNotificationStatus.Ready
import ru.yandex.direct.dbutil.SqlUtils
import ru.yandex.direct.dbutil.sharding.ShardHelper
import ru.yandex.direct.dbutil.sharding.ShardKey
import ru.yandex.direct.dbutil.wrapper.DslContextProvider
import ru.yandex.direct.env.NonDevelopmentEnvironment
import ru.yandex.direct.jobs.adfox.messaging.ytutils.YtReadException
import ru.yandex.direct.jobs.campdaybudgetlimitstoptime.QueueRecordParserFactory.createParser
import ru.yandex.direct.juggler.JugglerStatus
import ru.yandex.direct.juggler.check.annotation.JugglerCheck
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification
import ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_1_NOT_READY
import ru.yandex.direct.juggler.check.model.CheckTag.YT
import ru.yandex.direct.juggler.check.model.NotificationRecipient
import ru.yandex.direct.scheduler.Hourglass
import ru.yandex.direct.scheduler.support.DirectJob
import ru.yandex.direct.utils.DateTimeUtils
import ru.yandex.direct.utils.InterruptedRuntimeException
import ru.yandex.direct.ytwrapper.YtTableUtils
import ru.yandex.direct.ytwrapper.client.YtProvider
import ru.yandex.direct.ytwrapper.model.YtCluster
import ru.yandex.yt.ytclient.proxy.YtClient
import ru.yandex.yt.ytclient.tables.TableSchema
import ru.yandex.yt.ytclient.wire.UnversionedRow
import java.time.Duration.ofSeconds
import java.time.LocalDateTime
import java.util.function.Function
import javax.annotation.ParametersAreNonnullByDefault

/**
 * Описание параметров транспорта из БК.
 */
class CampDayBudgetProperties(config: DirectConfig) {
    /**
     * Описание параметров очереди, реализованной в виде упорядоченной таблицы YT
     */
    val ytCluster: YtCluster = YtCluster.valueOf(config.getString("queue_cluster").uppercase())
    val ytPath: String = config.getString("queue_path")
}

/**
 * Типизированное представление строки с сообщением из упорядоченной таблицы YT.
 */
data class QueueRecord(
    val rowIndex: Long,
    val orderId: Long,
    val stopTime: Long
)

/**
 * Парсер для запросов к упорядоченной таблице с сообщениями.
 */
object QueueRecordParserFactory {
    const val ROW_INDEX_COLUMN_NAME = "\$row_index"
    const val ORDER_ID_COLUMN_NAME = "OrderID"
    const val STOP_TIME_COLUMN_NAME = "StopTime"

    fun createParser(tableSchema: TableSchema): Function<UnversionedRow, QueueRecord> {
        val rowIndexIdx = YtTableUtils.findColumnOrThrow(tableSchema, ROW_INDEX_COLUMN_NAME)
        val orderIdIdx = YtTableUtils.findColumnOrThrow(tableSchema, ORDER_ID_COLUMN_NAME)
        val stopTimeIdx = YtTableUtils.findColumnOrThrow(tableSchema, STOP_TIME_COLUMN_NAME)
        return Function {
            val values = it.values
            val rowIndex = values[rowIndexIdx].longValue()
            val orderId = values[orderIdIdx].longValue()
            val stopTime = values[stopTimeIdx].longValue()
            QueueRecord(rowIndex, orderId, stopTime)
        }
    }
}

@Component
@Scope(SCOPE_PROTOTYPE)
class DayBudgetStopTimeDbHelper(
    private val dslContextProvider: DslContextProvider,
) {
    private var queries = mutableListOf<Query>()

    fun addUpdateStopTimeAndStatus(stopTime: Long, stopTimeLdt: LocalDateTime, campaignId: Long) {
        val timeField = if (stopTime == 0L) SqlUtils.mysqlZeroLocalDateTime() else DSL.value(stopTimeLdt)
        queries.add(
            DSL.update(CAMP_OPTIONS)
                .set(CAMP_OPTIONS.DAY_BUDGET_STOP_TIME, timeField)
                .set(CAMP_OPTIONS.DAY_BUDGET_NOTIFICATION_STATUS, Ready)
                .where(
                    CAMP_OPTIONS.CID.eq(campaignId)
                        .and(CAMP_OPTIONS.DAY_BUDGET_STOP_TIME.ne(timeField))
                )
        )
    }

    fun addInsertStopHistory(campaignId: Long, stopTimeLdt: LocalDateTime) =
        queries.add(
            DSL.insertInto(
                CAMP_DAY_BUDGET_STOP_HISTORY,
                CAMP_DAY_BUDGET_STOP_HISTORY.CID, CAMP_DAY_BUDGET_STOP_HISTORY.STOP_TIME
            ).values(campaignId, stopTimeLdt)
                .onDuplicateKeyIgnore()
        )

    fun executeTransactionBatch(shard: Int) {
        dslContextProvider.ppc(shard).transaction { configuration: Configuration ->
            queries.forEach { it.attach(configuration) }
            configuration.dsl().batch(queries).execute()
        }
        queries.clear()
    }
}

/**
 * Приём от БК информации о времени остановки заказов (кампаний) по дневному ограничению бюджета
 * Данные поступают в очередь (упорядоченную динамическую таблицу).
 * В очереди лежат идентификатор заказа, unix-время остановки кампании в случае остановки показов,
 * либо 0 в случае возобновления показов.
 */
@JugglerCheck(
    ttl = JugglerCheck.Duration(minutes = 15),
    needCheck = NonDevelopmentEnvironment::class,
    tags = [DIRECT_PRIORITY_1_NOT_READY, YT],
    notifications = [OnChangeNotification(
        recipient = [NotificationRecipient.LOGIN_IAM1], // https://st.yandex-team.ru/DIRECT-149028
        method = [NotificationMethod.TELEGRAM],
        status = [JugglerStatus.OK, JugglerStatus.CRIT]
    )]
)
@Hourglass(periodInSeconds = 120, needSchedule = NonDevelopmentEnvironment::class)
@ParametersAreNonnullByDefault
class SetCampDayBudgetLimitStopTimeJob(
    directConfig: DirectConfig,
    ppcPropertiesSupport: PpcPropertiesSupport,
    private val ytProvider: YtProvider,
    private val shardHelper: ShardHelper,
    private val campaignRepository: CampaignRepository,
    private val dayBudgetStopTimeDbHelper: DayBudgetStopTimeDbHelper
) : DirectJob() {

    companion object {
        private val logger = LoggerFactory.getLogger(SetCampDayBudgetLimitStopTimeJob::class.java)

        /**
         * Ключ, по которому хранится offset последнего прочитанного и обработанного сообщения
         */
        val LAST_READ_ROW_INDEX =
            PpcPropertyName("campdaybudgetlimitstoptime_last_read_row_index", PpcPropertyType.LONG)

        const val CONFIG_BRANCH_NAME = "camp_day_budget_limit_stop_time"
        private const val SELECT_ROWS_TIME_LIMIT_SEC: Long = 15
        private const val QUEUE_TABLET_INDEX = 0
        private const val QUEUE_BATCH_SIZE = 100
        private const val QUEUE_ARCHIVE_ROWS_SIZE: Long = 600000
    }

    private val jobConfig = CampDayBudgetProperties(directConfig.getBranch(CONFIG_BRANCH_NAME))
    private val lastReadRowIndexProperty = ppcPropertiesSupport[LAST_READ_ROW_INDEX]

    /**
     * Должна ли джоба писать изменения в базу (вместо perl), или же только помечать прочитанное
     */
    private val javaWritingEnabledProperty = ppcPropertiesSupport[CAMP_DAY_BUDGET_LIMIT_STOP_TIME_JAVA_WRITING_ENABLED]

    override fun execute() {
        val lastReadRowIndex = lastReadRowIndexProperty.getOrDefault(-1L)
        var newLastReadRowIndex: Long? = null
        var records: List<QueueRecord>
        do {
            records = read(newLastReadRowIndex ?: lastReadRowIndex)
            logger.info("got {} records", records.size)

            if (javaWritingEnabledProperty.getOrDefault(false)) {
                shardHelper.groupByShard(records, ShardKey.ORDER_ID) { it.orderId }
                    .forEach { shard, chunk ->
                        doDbUpdates(shard, chunk)
                    }
            }

            newLastReadRowIndex = records.lastOrNull()?.rowIndex
        } while (records.size == QUEUE_BATCH_SIZE)

        if (newLastReadRowIndex != null) {
            logger.info("newLastReadRowIndex = $newLastReadRowIndex")
            lastReadRowIndexProperty.set(newLastReadRowIndex)
            trimQueue(newLastReadRowIndex)
        }
    }

    /**
     * Вычитывает строки после позиции `lastReadRowIndex`.
     */
    private fun read(lastReadRowIndex: Long): List<QueueRecord> {
        val rowFrom = lastReadRowIndex + 1
        val rowTo = lastReadRowIndex + QUEUE_BATCH_SIZE
        val query = "* from [${jobConfig.ytPath}] where [\$tablet_index] = $QUEUE_TABLET_INDEX" +
            " and [\$row_index] between $rowFrom and $rowTo"
        logger.debug("Will query rows from {} by query: {}", jobConfig.ytPath, query)
        return try {
            val rowset = ytProvider.getDynamicOperator(jobConfig.ytCluster)
                .selectRows(query, ofSeconds(SELECT_ROWS_TIME_LIMIT_SEC))
            val rowConverter = createParser(rowset.schema)
            rowset.rows.map { rowConverter.apply(it) }
        } catch (e: Exception) {
            when (e) {
                is InterruptedException -> {
                    Thread.currentThread().interrupt()
                    throw InterruptedRuntimeException(e)
                }
                else ->
                    throw YtReadException("Can't get rows from ${jobConfig.ytPath}", e)
            }
        }
    }

    private fun doDbUpdates(
        shard: Int,
        chunk: List<QueueRecord>
    ) {
        val oid2cid = campaignRepository.getCidsForOrderIds(shard, chunk.map { it.orderId })
        chunk.filter { oid2cid[it.orderId] != null }
            .forEach { (rowIndex, orderId, stopTime) ->
                val stopTimeLdt = DateTimeUtils.fromEpochSeconds(stopTime)
                logger.debug(
                    "shard = {}, \$row_index = {}, order_id = {}, stop_time = {}, stopTimeLdt = {}",
                    shard, rowIndex, orderId, stopTime, stopTimeLdt
                )
                dayBudgetStopTimeDbHelper.addUpdateStopTimeAndStatus(stopTime, stopTimeLdt, oid2cid[orderId]!!)
                if (stopTime > 0) {
                    dayBudgetStopTimeDbHelper.addInsertStopHistory(oid2cid[orderId]!!, stopTimeLdt)
                }
            }
        dayBudgetStopTimeDbHelper.executeTransactionBatch(shard)
    }

    private fun trimQueue(newLastReadRowIndex: Long) {
        val yt = ytProvider.getDynamicOperator(jobConfig.ytCluster)
        val trimBeforeIndex = newLastReadRowIndex + 1 - QUEUE_ARCHIVE_ROWS_SIZE
        logger.info("trim table before {}", trimBeforeIndex)
        yt.runRpcCommandWithTimeout {
            yt.ytClient.trimTable(jobConfig.ytPath, QUEUE_TABLET_INDEX, trimBeforeIndex)
        }
    }
}
