package ru.yandex.direct.jobs.campqueue;

import java.time.Duration;
import java.time.LocalDateTime;
import java.time.temporal.ChronoUnit;
import java.util.List;
import java.util.concurrent.atomic.AtomicLong;

import org.jooq.util.mysql.MySQLDSL;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.common.db.PpcPropertyNames;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.env.NonDevelopmentEnvironment;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.annotation.Url;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.HourglassDaemon;
import ru.yandex.direct.scheduler.support.DirectShardedJob;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.registry.MetricRegistry;

import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMP_OPERATIONS_QUEUE_COPY;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_0;
import static ru.yandex.direct.solomon.SolomonUtils.SOLOMON_REGISTRY;

/**
 * Отправляет в соломон информацию о состоянии очереди camp_operations_queue_copy:
 * возраст самого старого объекта (в часах), размер очереди и количество уникальных клиентов,
 * ожидающих копирования. Список клиентов так же логирует (может быть урезанным).
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(minutes = 10),
        tags = {DIRECT_PRIORITY_0},
        needCheck = NonDevelopmentEnvironment.class,
        urls = {
                @Url(title = "Возраст очереди", url = CampQueueMonitoringJob.URL_SOLOMON_HOURS),
                @Url(title = "Размер очереди", url = CampQueueMonitoringJob.URL_SOLOMON_SIZE),
                @Url(title = "Количество ожидающих клиентов", url = CampQueueMonitoringJob.URL_SOLOMON_CLIENTS),
                @Url(title = "Дока на скрипт копирования",
                        url = "https://docs.yandex-team.ru/direct-dev/reference/perl-scripts/list/ppcCampQueue.pl"),
        }
)
@HourglassDaemon(runPeriod = 60)
@Hourglass(periodInSeconds = 60, needSchedule = ProductionOnly.class)
public class CampQueueMonitoringJob extends DirectShardedJob {
    private static final Logger logger = LoggerFactory.getLogger(CampQueueMonitoringJob.class);

    static final String URL_SOLOMON_HOURS = "https://solomon.yandex-team.ru/?project=direct&cluster=app_java-jobs" +
            "&service=java_jobs&l.sensor=age_minutes&l.host=CLUSTER&graph=auto&l.queue=camp_operations_queue_copy" +
            "&l.env=production&stack=false";
    static final String URL_SOLOMON_SIZE = "https://solomon.yandex-team.ru/?project=direct&cluster=app_java-jobs" +
            "&service=java_jobs&l.sensor=queue_size&l.host=CLUSTER&graph=auto&l.queue=camp_operations_queue_copy" +
            "&l.env=production&stack=false";
    static final String URL_SOLOMON_CLIENTS = "https://solomon.yandex-team.ru/?project=direct&cluster=app_java-jobs" +
            "&service=java_jobs&l.sensor=clients_waiting&l.host=CLUSTER&graph=auto&l.queue=camp_operations_queue_copy" +
            "&l.env=production&stack=false";

    private static final String AGE_MINUTES = "age_minutes";
    private static final String QUEUE_SIZE = "queue_size";
    private static final String CLIENTS_WAITING = "clients_waiting";

    private final AtomicLong ageMinutes = new AtomicLong();
    private final AtomicLong queueSize = new AtomicLong();
    private final AtomicLong clientsWaiting = new AtomicLong();

    private static final long AGE_LIMIT_HOURS_DEFAULT = 4;

    private static final int CLIENTS_OUTPUT_LIMIT = 1000;

    private final DslContextProvider dslContextProvider;
    private final PpcProperty<Long> limitHoursProperty;

    private MetricRegistry metricRegistry;

    @Autowired
    public CampQueueMonitoringJob(DslContextProvider dslContextProvider,
                                  PpcPropertiesSupport ppcPropertiesSupport) {
        this.dslContextProvider = dslContextProvider;
        this.limitHoursProperty =
                ppcPropertiesSupport.get(PpcPropertyNames.CAMP_OPERATIONS_QUEUE_COPY_AGE_LIMIT_HOURS);
    }

    @Override
    public void execute() {
        this.metricRegistry = lazyInitMetricRegistry();

        ageMinutes.set(getAgeMinutes());
        queueSize.set(getQueueSize());
        clientsWaiting.set(getClientsWaiting());

        setStatus();
    }

    private MetricRegistry lazyInitMetricRegistry() {
        Labels solomonRegistryLabels =
                Labels.of("queue", "camp_operations_queue_copy", "shard", String.valueOf(getShard()));
        MetricRegistry metricRegistry = SOLOMON_REGISTRY.subRegistry(solomonRegistryLabels);
        metricRegistry.lazyGaugeInt64(AGE_MINUTES, ageMinutes::get);
        metricRegistry.lazyGaugeInt64(QUEUE_SIZE, queueSize::get);
        metricRegistry.lazyGaugeInt64(CLIENTS_WAITING, clientsWaiting::get);
        return metricRegistry;
    }

    private long getAgeMinutes() {
        LocalDateTime oldestRecord = dslContextProvider.ppc(getShard())
                .select(MySQLDSL.min(CAMP_OPERATIONS_QUEUE_COPY.QUEUE_TIME))
                .from(CAMP_OPERATIONS_QUEUE_COPY)
                .fetchOne()
                .value1();

        if (oldestRecord == null) {
            return 0;
        }

        LocalDateTime now = LocalDateTime.now();

        long ageMinutes = ChronoUnit.MINUTES.between(oldestRecord, now);
        logger.info("Queue age: {} minutes", ageMinutes);
        return ageMinutes;
    }

    private long getQueueSize() {
        int queueSize = dslContextProvider.ppc(getShard())
                .selectCount()
                .from(CAMP_OPERATIONS_QUEUE_COPY)
                .fetchOne()
                .value1();

        if (queueSize > 0) {
            logger.info("Queue_is_not_empty");
        }

        logger.info("Queue size: {} records", queueSize);
        return queueSize;
    }

    private long getClientsWaiting() {
        List<Long> clientsWaiting = dslContextProvider.ppc(getShard())
                .selectDistinct(CAMPAIGNS.CLIENT_ID)
                .from(CAMP_OPERATIONS_QUEUE_COPY)
                .join(CAMPAIGNS).on(CAMPAIGNS.CID.eq(CAMP_OPERATIONS_QUEUE_COPY.CID))
                .fetch(CAMPAIGNS.CLIENT_ID);

        int clientsWaitingOriginalSize = clientsWaiting.size();

        if (clientsWaiting.size() > CLIENTS_OUTPUT_LIMIT) {
            clientsWaiting = clientsWaiting.subList(0, CLIENTS_OUTPUT_LIMIT);
        }

        logger.info("Clients waiting: {}", clientsWaiting);
        return clientsWaitingOriginalSize;
    }

    private void setStatus() {
        Duration ageThreshold = Duration.ofHours(limitHoursProperty.getOrDefault(AGE_LIMIT_HOURS_DEFAULT));
        long ageThresholdMinutes = ageThreshold.toMinutes();
        long currentAge = ageMinutes.get();
        if (currentAge > ageThresholdMinutes) {
            String desc = String.format("Queue age exceed threshold. Current: %dm. Threshold: %dm",
                    currentAge, ageThresholdMinutes);
            logger.info("Queue_age_exceeds_the_limit");
            setJugglerStatus(JugglerStatus.CRIT, desc);
        } else {
            String desc = String.format("Queue age OK. Threshold: %dm", ageThresholdMinutes);
            setJugglerStatus(JugglerStatus.OK, desc);
        }
    }

    @Override
    public void finish() {
        metricRegistry.removeMetric(AGE_MINUTES, Labels.empty());
        metricRegistry.removeMetric(QUEUE_SIZE, Labels.empty());
        metricRegistry.removeMetric(CLIENTS_WAITING, Labels.empty());
    }
}
