package ru.yandex.direct.jobs.canvas;

import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Sets;

import ru.yandex.direct.canvas.tools_client.CanvasToolsClient;
import ru.yandex.direct.core.entity.creative.repository.CreativeRepository;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.env.TypicalEnvironment;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.model.CheckTag;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectShardedJob;

import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_2;

/**
 * Джоба для фонового переснятия скриншотов креативов, для которых ротор с первого раза не смог снять скриншот.
 * с помощью {@link ru.yandex.direct.canvas.tools_client.CanvasToolsClient}.
 * Использует те же ручки в канвасе, что и https://test-direct.yandex.ru/internal_tools/#canvas_on_creatives_operation.
 *
 * Нужна чтобы при открытии гридов баннеров скриншоты креатива уже были готовы и быстрее подгружались.
 * Джоба не критичная. Если не будет работать, то эти скриншоты доснимутся во время открытия грида баннеров.
 */
@JugglerCheck(
        ttl = @JugglerCheck.Duration(hours = 4),
        tags = {DIRECT_PRIORITY_2, CheckTag.DIRECT_PRODUCT_TEAM},
        needCheck = ProductionOnly.class
)
@Hourglass(periodInSeconds = 900, needSchedule = TypicalEnvironment.class)
@ParametersAreNonnullByDefault
public class CanvasScreenshotJob extends DirectShardedJob {
    private static final int DB_CHUNK_SIZE = 3;
    private final CreativeRepository creativeRepository;
    private final CanvasToolsClient canvasToolsClient;

    public CanvasScreenshotJob(CreativeRepository creativeRepository, CanvasToolsClient canvasToolsClient) {
        this.creativeRepository = creativeRepository;
        this.canvasToolsClient = canvasToolsClient;
    }

    @Override
    public void execute() {
        Set<Long> creativeIds = Sets.newHashSet(
                creativeRepository.getCreativeIdsWithoutScreenshoot(getShard(), DB_CHUNK_SIZE));
        if (creativeIds.isEmpty()) {
            return;
        }
        canvasToolsClient.reshootScreenshot(creativeIds);
        canvasToolsClient.sendToDirect(creativeIds);
    }
}
