package ru.yandex.direct.jobs.canvas;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;


import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod;
import ru.yandex.direct.canvas.tools_client.CanvasToolsClient;
import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.env.TypicalEnvironment;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification;
import ru.yandex.direct.juggler.check.model.NotificationRecipient;
import ru.yandex.direct.liveresource.LiveResourceFactory;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectJob;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YqlQuery;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.yql.ResultSetFuture;
import ru.yandex.yql.YqlConnection;

import static ru.yandex.direct.common.db.PpcPropertyNames.CMS_VIDEO_RETRANSCODE_BATCH_SIZE;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_2;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.ytwrapper.model.YtSQLSyntaxVersion.SQLv1;

@ParametersAreNonnullByDefault
@JugglerCheck(
        ttl = @JugglerCheck.Duration(hours = 72),
        tags = {DIRECT_PRIORITY_2},
        needCheck = ProductionOnly.class,
        notifications = @OnChangeNotification(
            recipient = NotificationRecipient.LOGIN_ALEX_KULAKOV,
            method = NotificationMethod.TELEGRAM,
            status = {JugglerStatus.OK, JugglerStatus.CRIT}
        )
)
@Hourglass(cronExpression = "0 0 1,4,7 * * ?", needSchedule = TypicalEnvironment.class)
public class ConvertVideoToNewFormatJob extends DirectJob {

    private static final Logger logger = LoggerFactory.getLogger(ConvertVideoToNewFormatJob.class);

    private static final String REQUEST_PATH = "classpath:///convervideotonewformat/convervideotonewformat.yql";

    private final String query;
    private final YtProvider ytProvider;

    private final PpcPropertiesSupport ppcPropertiesSupport;

    private final CanvasToolsClient canvasToolsClient;


    public ConvertVideoToNewFormatJob(YtProvider ytProvider, PpcPropertiesSupport ppcPropertiesSupport,
                                      CanvasToolsClient canvasToolsClient) {
        this.ytProvider = ytProvider;
        this.query = LiveResourceFactory.get(REQUEST_PATH).getContent();
        this.canvasToolsClient = canvasToolsClient;
        this.ppcPropertiesSupport = ppcPropertiesSupport;
    }

    ResultSetFuture runQuery(int numberVideosToConvert) throws SQLException {
        var ytOperator = this.ytProvider.getOperator(YtCluster.HAHN);

        logger.info("Running YQL.");
        var startingQuery = ytOperator.yqlQueryBegin(new YqlQuery(this.query, numberVideosToConvert));
        return (ResultSetFuture) checkOperation(startingQuery.getOperationId());
    }


    private void convertVideo(ResultSetFuture resultSetFuture) throws ExecutionException, InterruptedException,
            SQLException {
        int countConvertedVideos  = 0;
        try {
            ResultSet rs = resultSetFuture.get();
            logger.info("YQL completed.");

            while (rs.next()) {
                String videoId = rs.getString("video_id");
                canvasToolsClient.cmsRecode(videoId);
                logger.info("Video with id = {} in queue for conversion.", videoId);
                countConvertedVideos++;
            }

        } catch (InterruptedException | SQLException | ExecutionException e) {
            logger.error("Some exception occured.", e);
            throw e;
        }
        logger.info("{} videos in queue for conversion.", countConvertedVideos);
    }

    Future<?> checkOperation(String operationId) throws SQLException {
        YqlConnection connection = (YqlConnection) ytProvider.getYql(YtCluster.HAHN, SQLv1).getConnection();
        return connection.restoreResultSetFuture(operationId, "");
    }

    @Override
    public void execute() {
        var numberVideosToConvert = ppcPropertiesSupport.get(CMS_VIDEO_RETRANSCODE_BATCH_SIZE).get();
        if (nvl(numberVideosToConvert, 0) == 0) {
            logger.info("Skip job iteration CMS_VIDEO_RETRANSCODE_BATCH_SIZE=0.");
        } else {
            try {
                var resultSetFuture = runQuery(numberVideosToConvert);
                convertVideo(resultSetFuture);
            } catch (InterruptedException | SQLException | ExecutionException e) {
                logger.error("Сonversion failed.");
            }
        }
    }
}
