package ru.yandex.direct.jobs.canvasoperationsoncreatives;

import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.canvas.model.OnCreativeOperationResult;
import ru.yandex.direct.canvas.model.OnCreativeOperationResultStatus;
import ru.yandex.direct.canvas.tools_client.CanvasToolsClient;
import ru.yandex.direct.core.entity.creative.repository.CreativeRepository;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.env.TypicalEnvironment;
import ru.yandex.direct.jobs.canvasoperationsoncreatives.model.OperationName;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.model.CheckTag;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectJob;
import ru.yandex.direct.tracing.Trace;

import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_2;

/**
 * Джоба для массового выполнения операций из {@link ru.yandex.canvas.controllers.InternalToolsController}
 * с помощью {@link ru.yandex.direct.canvas.tools_client.CanvasToolsClient}.
 * Использует те же ручки в канвасе, что и https://test-direct.yandex.ru/internal_tools/#canvas_on_creatives_operation.
 * <p>
 * Может использоваться для запуска операций
 * {@link ru.yandex.direct.jobs.canvasoperationsoncreatives.model.OperationName}
 * над всеми креативами или над диапазоном идентификаторов креативов.
 * При этом самостоятельно отфильтровыввает креативы, созданные на основе стоковых.
 * <p>
 * Для запуска на CPM_VIDEO_CREATIVE креативах достаточно прописать в ppc_properties
 * JOBS_CANVAS_OPERATIONS_ON_CREATIVES_RANGES = {"operation_names":["rebuild"],"creative_types":["CPM_VIDEO_CREATIVE"]}.
 * Для более тонкой настройки можно заполнить диапазоны идентификаторов креативов в ppc_properties
 * отдельно для каждого шарда, см.
 * {@link ru.yandex.direct.jobs.canvasoperationsoncreatives.model.VideoCreativesRangesList}.
 * Пример для обработки всех креативов на 1м шарде:
 * {"ranges":[{"shard_id":1, "start_id":0}], "operation_names":["rebuild"],"creative_types":["CPM_VIDEO_CREATIVE"]}
 * В процессе работы джоба обновляет запись в ppc_properties и таким оразом фиксирует прогресс обработки креативов.
 * Операции выполняются в том порядке, в котором заданы в массиве "operation_names".
 * <p>
 * Для остановки джобы достаточно записать в JOBS_CANVAS_OPERATIONS_ON_CREATIVES_RANGES значение
 * с пустым массивом operation_names
 */
@JugglerCheck(
        ttl = @JugglerCheck.Duration(hours = 1),
        tags = {DIRECT_PRIORITY_2, CheckTag.DIRECT_PRODUCT_TEAM},
        needCheck = ProductionOnly.class
)
@Hourglass(periodInSeconds = 30, needSchedule = TypicalEnvironment.class)
@ParametersAreNonnullByDefault
public class CanvasOperationsOnCreativesJob extends DirectJob {
    private static final Logger logger = LoggerFactory.getLogger(CanvasOperationsOnCreativesJob.class);
    private static final String TRACE_FUNC = "canvas_operations_job";

    private final VideoCreativesRangeService videoCreativesRangeService;
    private final CanvasToolsClient canvasToolsClient;
    private final CreativeRepository creativeRepository;

    public CanvasOperationsOnCreativesJob(
            VideoCreativesRangeService videoCreativesRangeService,
            CanvasToolsClient canvasToolsClient,
            CreativeRepository creativeRepository
    ) {
        this.videoCreativesRangeService = videoCreativesRangeService;
        this.canvasToolsClient = canvasToolsClient;
        this.creativeRepository = creativeRepository;
    }

    @Override
    public void execute() {
        var rangesStringValue = videoCreativesRangeService.getStringValue();
        var ranges = videoCreativesRangeService.parse(rangesStringValue);
        if (ranges == null) {
            logger.warn("Empty or invalid video creatives ranges, job will not start");
            return;
        }

        for (var range : ranges.getRanges()) {
            if (range.getFinished() != null && range.getFinished()) {
                logger.info("Job is done for shard {}", range.getShardId());
                continue;
            }

            List<Long> creativeIds;
            try (var ignore = Trace.current().profile(TRACE_FUNC, "get_all_creative_ids")) {
                creativeIds = creativeRepository.getAllCreativeIds(
                        range.getShardId(),
                        range.getStartId(),
                        range.getEndId(),
                        ranges.getCreativeTypes(),
                        ranges.getDbChunkSize());
            }

            applyOperations(creativeIds, ranges.getOperationNames(), ranges.getRequestChunkSize());
            videoCreativesRangeService.extend(range, creativeIds);
        }

        videoCreativesRangeService.save(rangesStringValue, ranges);
    }

    private void applyOperations(List<Long> creativeIds, List<OperationName> operationNames, Integer requestChunkSize) {
        for (List<Long> idsChunk : Lists.partition(creativeIds, requestChunkSize)) {
            for (var operation : operationNames) {
                try {
                    Map<Long, OnCreativeOperationResult> result;
                    try (var ignore = Trace.current().profile(TRACE_FUNC, operation.getName().toLowerCase())) {
                        result = executeOperation(operation, Sets.newHashSet(idsChunk));
                    }

                    var failedCreatives = getFailedCreatives(result);
                    if (!failedCreatives.isEmpty()) {
                        logger.warn("Failed to perform {} operation on creatives {}", operation, failedCreatives);
                    }
                } catch (Exception ex) {
                    logger.error(
                            "Failed to perform {} operation on creatives {}, ex - {}", operation, idsChunk, ex);
                }
            }
        }
    }

    private Map<Long, OnCreativeOperationResult> executeOperation(OperationName operation, Set<Long> creativeIds) {
        switch (operation) {
            case RESHOOT_SCREENSHOT:
                return canvasToolsClient.reshootScreenshot(creativeIds);
            case REBUILD:
                return canvasToolsClient.rebuild(creativeIds);
            case SEND_TO_DIRECT:
                return canvasToolsClient.sendToDirect(creativeIds);
            case SEND_TO_RTBHOST:
                return canvasToolsClient.sendToRtbHost(creativeIds);
            default:
                throw new IllegalStateException("Unexpected enum value: " + operation + " with no handler");
        }
    }

    private List<Long> getFailedCreatives(Map<Long, OnCreativeOperationResult> resultMap) {
        return resultMap.keySet()
                .stream()
                .filter(id -> OnCreativeOperationResultStatus.ERROR.equals(resultMap.get(id).getStatus()))
                .collect(toList());
    }
}
