package ru.yandex.direct.jobs.canvasoperationsoncreatives;

import java.util.List;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Strings;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.jobs.canvasoperationsoncreatives.model.VideoCreativesRange;
import ru.yandex.direct.jobs.canvasoperationsoncreatives.model.VideoCreativesRangesList;

import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.core.entity.ppcproperty.model.PpcPropertyEnum.JOBS_CANVAS_OPERATIONS_ON_CREATIVES_RANGES;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.JsonUtils.fromJson;
import static ru.yandex.direct.utils.JsonUtils.toJson;

@Component
@ParametersAreNonnullByDefault
public class VideoCreativesRangeService {
    private static final Logger logger = LoggerFactory.getLogger(VideoCreativesRangeService.class);
    private static final int DEFAULT_DB_CHUNK_SIZE = 200;
    private static final int DEFAULT_REQUEST_CHUNK_SIZE = 20;

    private final PpcPropertiesSupport ppcPropertiesSupport;
    private final ShardHelper shardHelper;

    public VideoCreativesRangeService(
            PpcPropertiesSupport ppcPropertiesSupport,
            ShardHelper shardHelper
    ) {
        this.ppcPropertiesSupport = ppcPropertiesSupport;
        this.shardHelper = shardHelper;
    }

    public String getStringValue() {
        return ppcPropertiesSupport.get(JOBS_CANVAS_OPERATIONS_ON_CREATIVES_RANGES.getName());
    }

    public VideoCreativesRangesList parse(@Nullable String rangesStringValue) {
        if (Strings.isNullOrEmpty(rangesStringValue)) {
            logger.warn(
                    "ppc_properties value not found for property name {}",
                    JOBS_CANVAS_OPERATIONS_ON_CREATIVES_RANGES.getName());
            return null;
        }

        try {
            var ranges = fromJson(rangesStringValue, VideoCreativesRangesList.class);
            if (ranges.getOperationNames().isEmpty()) {
                logger.info("Operation list is empty");
                return null;
            }

            if (ranges.getCreativeTypes().isEmpty()) {
                logger.info("Creative types list is empty");
                return null;
            }

            // init ranges to simplify job's setup
            if (ranges.getRanges() == null || ranges.getRanges().isEmpty()) {
                ranges.setRanges(shardHelper.dbShards()
                        .stream()
                        .map(shard -> new VideoCreativesRange(shard, 0L, null, false))
                        .collect(toList()));
            }

            ranges.setDbChunkSize(nvl(ranges.getDbChunkSize(), DEFAULT_DB_CHUNK_SIZE));
            ranges.setRequestChunkSize(nvl(ranges.getRequestChunkSize(), DEFAULT_REQUEST_CHUNK_SIZE));
            return ranges;
        } catch (IllegalArgumentException ex) {
            logger.error("Invalid video creatives ranges json " + rangesStringValue, ex);
            return null;
        }
    }

    public void extend(VideoCreativesRange range, List<Long> creativeIds) {
        if (creativeIds.isEmpty()) {
            logger.info("Didn't find creatives, range is set to finished {}", range);
            range.setFinished(true);
            return;
        }

        range.setStartId(creativeIds.stream().max(Long::compareTo).get() + 1);
        logger.info("Video creatives range extended, {}", range);
    }

    public void save(String rangesStringValue, VideoCreativesRangesList ranges) {
        try {
            ppcPropertiesSupport.cas(
                    JOBS_CANVAS_OPERATIONS_ON_CREATIVES_RANGES.getName(), rangesStringValue, toJson(ranges));
        } catch (Exception ex) {
            logger.error("Couldn't save video creatives ranges, you can update them manually " + ranges, ex);
        }
    }
}
