package ru.yandex.direct.jobs.cashback;

import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.core.entity.cashback.model.CashbackRewardsImportParams;
import ru.yandex.direct.core.entity.cashback.service.ImportCashbackRewardsService;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.jobs.abt.check.TableExistsChecker;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectJob;
import ru.yandex.direct.ytwrapper.YtPathUtil;
import ru.yandex.direct.ytwrapper.model.YtCluster;

import static ru.yandex.direct.common.db.PpcPropertyNames.IMPORT_CASHBACK_REWARDS_DETAILS_ENABLED;
import static ru.yandex.direct.common.db.PpcPropertyNames.LAST_IMPORTED_CASHBACK_TABLE;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_1;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Джоб для импорта детализации начисленного клиентам кешбэка.
 * <p>
 * Раз в месяц создаётся новая таблица в YT с данными за предыдущий месяц,
 * эти данные надо забрать и положить у нас в MySQL для отображения клиентам.
 * <p>
 * Джоб при этом запускается ежедневно, чтобы не привязываться к конкретной дате
 * появления новой таблицы и отслеживать работоспособность джобы.
 * <p>
 * Важно: У импортируемых данных нет тестовой версии (отдельной тестовой директории
 * в YT), поэтому при тестировании используются продовые данные, из-за чего
 * крайне желательно перед запуском на тестовых средах сузить количество импортируемых
 * месяцев с помощью соответствующего
 * {@link ru.yandex.direct.common.db.PpcPropertyNames#LAST_IMPORTED_CASHBACK_TABLE свойства}.
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(days = 1, hours = 12), needCheck = ProductionOnly.class, tags =
        {DIRECT_PRIORITY_1})
@Hourglass(periodInSeconds = 60 * 60, needSchedule = ProductionOnly.class)
@ParametersAreNonnullByDefault
public class ImportCashbackRewardsDetailsJob extends DirectJob {
    private static final String DEFAULT_LAST_IMPORTED_TABLE = "202007";
    private static final DateTimeFormatter DATE_FORMAT = DateTimeFormatter.ofPattern("yyyyMM");

    private static final Logger logger = LoggerFactory.getLogger(ImportCashbackRewardsDetailsJob.class);

    private final TableExistsChecker tableExistsChecker;
    private final PpcPropertiesSupport ppcPropertiesSupport;
    private final ImportCashbackRewardsService rewardsService;

    private final List<YtCluster> clusters;
    private final String baseYtPath;

    @Autowired
    public ImportCashbackRewardsDetailsJob(TableExistsChecker tableExistsChecker,
                                           PpcPropertiesSupport ppcPropertiesSupport,
                                           ImportCashbackRewardsService rewardsService,
                                           DirectConfig directConfig) {
        this.tableExistsChecker = tableExistsChecker;
        this.ppcPropertiesSupport = ppcPropertiesSupport;
        this.rewardsService = rewardsService;

        var cfg = directConfig.getBranch("cashbacks.yt");

        var clusterNames = cfg.getStringList("clusters");
        clusters = mapList(clusterNames, YtCluster::parse);
        baseYtPath = cfg.getString("rewards_table_base_path");
    }

    @Override
    public void execute() {
        logger.info("START");

        var isJobEnabled = ppcPropertiesSupport.get(IMPORT_CASHBACK_REWARDS_DETAILS_ENABLED)
                .getOrDefault(false);
        if (!isJobEnabled) {
            logger.info("Import is disabled by property now, finish");
            return;
        }

        var lastImportedTableName = ppcPropertiesSupport.get(LAST_IMPORTED_CASHBACK_TABLE)
                .getOrDefault(DEFAULT_LAST_IMPORTED_TABLE);
        var monthToSearchFrom = getDateByTableName(lastImportedTableName);
        var previousMonthDate = LocalDate.now().withDayOfMonth(1).minusMonths(1L);
        var tablesByDate = getTablesToImport(monthToSearchFrom, previousMonthDate, lastImportedTableName);
        logger.info("Found " + tablesByDate.size() + " months to import");
        tablesByDate.forEach(rewardsService::importRewardsTables);
        if (tablesByDate.size() > 0) {
            var lastImportedMonth = tablesByDate.get(tablesByDate.size() - 1);
            var lastImportedTable = lastImportedMonth.get(lastImportedMonth.size() - 1);
            ppcPropertiesSupport.get(LAST_IMPORTED_CASHBACK_TABLE).set(getTableNameFromPath(lastImportedTable.getTablePath()));
        }
        logger.info("END");
    }

    /**
     * Находит таблицы, которые надо импортировать
     * <p>
     * В YT каждый месяц раскладывается в основную таблицу внутри базовой директории с именем вида "yyyyMM".
     * Так же может быть несколько дополнительных таблиц с имнами вида "yyyyMM_i"
     * Исходя из этого, метод строит полные пути к таблицам, проверяет, что таблица по заданному пути
     * существует по порядку для каждого кластера из конфига (в порядке их определения), и добавляет
     * соответствующие параметры в результат. Если для месяца есть таблица в кластере, то следующие кластеры
     * проверяться не будут.
     *
     * @param fromDate дата, от которой искать таблицы
     * @param toDate   дата, до которой искать таблицы, включительно
     * @param lastImportedTable имя последней импортированной таблицы
     * @return список списков {@link CashbackRewardsImportParams параметров} для импорта таблиц для каждого месяца
     * (Для каждого месяца список таблиц для импорта)
     */
    private List<List<CashbackRewardsImportParams>> getTablesToImport(LocalDate fromDate, LocalDate toDate,
                                                                      String lastImportedTable) {
        List<List<CashbackRewardsImportParams>> result = new ArrayList<>();
        while (!fromDate.isAfter(toDate)) {
            for (var cluster : clusters) {
                var tables = getTablesFromClusterByDate(cluster, fromDate, lastImportedTable);
                if (!tables.isEmpty()) {
                    result.add(tables);
                    break;
                }
            }
            fromDate = fromDate.plusMonths(1);
        }
        return result;
    }

    /**
     * Находит таблицы (основную и дополнительные) в кластере для выбранного месяца
     * Основная таблица имеет имя вида "yyyyMM", дополнительные имеют имена вида "yyyyMM_i"
     * Метод проверет наличие основной таблицы, затем проверяет наличие дополнительных таблиц "yyyyMM_1", "yyyyMM_2", ...
     * Когда очередной дополнительной таблицы не оказывается, метод завершает работу и возвращает список найденных таблиц
     * для данного месяца
     * @param cluster YT-кластер, в котором искать таблицы
     * @param date дата, по которой искать таблицы
     * @param lastImportedTable имя последней импортированной таблицы
     * @return список {@link CashbackRewardsImportParams параметров} для импорта таблиц для этого месяца
     */
    private List<CashbackRewardsImportParams> getTablesFromClusterByDate(YtCluster cluster, LocalDate date, String lastImportedTable) {
        boolean haveNewTables = false;
        List<CashbackRewardsImportParams> result = new ArrayList<>();
        String tableName = date.format(DATE_FORMAT);
        String baseTablePath = YtPathUtil.generatePath(baseYtPath, tableName);
        if (tableExistsChecker.check(cluster, baseTablePath)) {
            if (lastImportedTable.compareTo(tableName) < 0) {
                //Have new table for this month
                haveNewTables = true;
            }
            result.add(new CashbackRewardsImportParams()
                    .withCluster(cluster)
                    .withDate(date)
                    .withTablePath(baseTablePath));
        } else {
            return List.of();
        }
        int additionalTableIndex = 1;
        while (true) {
            String additionalTableName = tableName + "_" + additionalTableIndex;
            String additionalTablePath = YtPathUtil.generatePath(baseYtPath, additionalTableName);
            additionalTableIndex += 1;
            if (tableExistsChecker.check(cluster, additionalTablePath)) {
                if (lastImportedTable.compareTo(additionalTableName) < 0) {
                    //Have new additional table for this month
                    haveNewTables = true;
                }
                result.add(new CashbackRewardsImportParams()
                        .withCluster(cluster)
                        .withDate(date)
                        .withTablePath(additionalTablePath));
            } else {
                break;
            }
        }
        return haveNewTables ? result : List.of();
    }

    private static LocalDate getDateByTableName(String tableName) {
        var year = Integer.parseInt(tableName.substring(0, 4));
        var month = Integer.parseInt(tableName.substring(4, 6));
        return LocalDate.of(year, month, 1);
    }

    private static String getTableNameFromPath(String path) {
        var splited = path.split("/");
        return splited[splited.length - 1];
    }
}
