package ru.yandex.direct.jobs.cashback;

import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.common.db.PpcPropertyNames;
import ru.yandex.direct.communication.service.CommunicationEventService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectJob;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.direct.ytwrapper.model.YtOperator;
import ru.yandex.direct.ytwrapper.model.YtTable;
import ru.yandex.inside.yt.kosher.cypress.CypressNodeType;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;

import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_1;

@JugglerCheck(ttl = @JugglerCheck.Duration(days = 1), needCheck = ProductionOnly.class, tags =
        {DIRECT_PRIORITY_1})
@Hourglass(periodInSeconds = 60 * 60, needSchedule = ProductionOnly.class)
public class NotifyClientCashbackJob extends DirectJob {

    private static final Logger logger = LoggerFactory.getLogger(NotifyClientCashbackJob.class);

    private static final String CREATION_TIME_ATTRIBUTE = "creation_time";
    private static final String NAME_ATTRIBUTE = "key";
    private static final String TYPE_ATTRIBUTE = "type";
    private static final String ROW_COUNT_ATTRIBUTE = "row_count";
    private static final SetF<String> ATTRIBUTES = Cf.set(
            NAME_ATTRIBUTE, CREATION_TIME_ATTRIBUTE, TYPE_ATTRIBUTE, ROW_COUNT_ATTRIBUTE);

    private final PpcProperty<String> clusterProperty;
    private final PpcProperty<String> directoryPathProperty;
    private final PpcProperty<String> tableCreationTimeProperty;
    private final PpcProperty<Integer> rowNumberProperty;
    private final PpcProperty<Integer> chunkSizeProperty;
    private final YtProvider ytProvider;
    private final CommunicationEventService eventService;

    @Autowired
    public NotifyClientCashbackJob(
            PpcPropertiesSupport ppcPropertiesSupport,
            YtProvider ytProvider,
            CommunicationEventService eventService
    ) {
        clusterProperty = ppcPropertiesSupport.get(PpcPropertyNames.NOTIFY_CASHBACK_CLUSTER);
        directoryPathProperty = ppcPropertiesSupport.get(PpcPropertyNames.NOTIFY_CASHBACK_DIRECTORY);
        tableCreationTimeProperty = ppcPropertiesSupport.get(PpcPropertyNames.NOTIFY_CASHBACK_TABLE_CREATTION_TIME);
        rowNumberProperty = ppcPropertiesSupport.get(PpcPropertyNames.NOTIFY_CASHBACK_ROW_NUMBER);
        chunkSizeProperty = ppcPropertiesSupport.get(PpcPropertyNames.NOTIFY_CASHBACK_CHUNK_SIZE);
        this.ytProvider = ytProvider;
        this.eventService = eventService;
    }

    private YtOperator getYtOperator() {
        var clusterName = clusterProperty.getOrDefault(YtCluster.HAHN.getName());
        logger.info("cluster: " + clusterName);
        return ytProvider.getOperator(YtCluster.parse(clusterName));
    }

    private List<YTreeNode> getTables(
            YtOperator ytOperator
    ) {
        var directory = directoryPathProperty.get();
        logger.info("directory: " + directory);
        if (directory == null) {
            return Collections.emptyList();
        }
        var lastHandledTableCreationTime = tableCreationTimeProperty.getOrDefault("0");
        logger.info("lastHandledTableCreationTime = " + lastHandledTableCreationTime);

        var cypress = ytOperator.getYt().cypress();
        var path = YPath.simple(directory);
        Collection<YTreeNode> nodes = cypress.get(path, ATTRIBUTES).asMap().values();
        var tables = StreamEx.of(nodes)
                .filter(node -> node.getAttribute(TYPE_ATTRIBUTE).get().stringValue()
                        .equals(CypressNodeType.TABLE.value()))
                .mapToEntry(table -> table.getAttribute(CREATION_TIME_ATTRIBUTE).get().stringValue())
                .filterValues(creationTime -> creationTime.compareTo(lastHandledTableCreationTime) > 0)
                .sortedBy(e -> e.getValue())
                .keys()
                .toList();
        logger.info("tables: " + tables);
        return tables;
    }

    @Override
    public void execute() {
        logger.info("START");
        var ytOperator = getYtOperator();
        getTables(ytOperator).forEach(table -> handleTable(table, ytOperator));
        logger.info("END");
    }

    public void handleTable(YTreeNode table, YtOperator ytOperator) {
        var rowCount = table.getAttribute(ROW_COUNT_ATTRIBUTE).get().intValue();
        var currentRow = rowNumberProperty.getOrDefault(0);
        YtTable ytTable = new YtTable(directoryPathProperty.get() + "/" +
                table.getAttribute(NAME_ATTRIBUTE).get().stringValue());
        while (currentRow < rowCount) {
            var chunkSize = chunkSizeProperty.getOrDefault(100);
            var endRow = Math.min(rowCount, currentRow + chunkSize);
            NotificationDataTableRow tableRow = new NotificationDataTableRow();
            Map<ClientId, Map<String, String>> paramsByClient = new HashMap<>(endRow - currentRow);
            ytOperator.readTableByRowRange(ytTable,
                    row -> paramsByClient.put(row.getClientId(), row.getParams()),
                    tableRow,
                    currentRow,
                    endRow);
            currentRow = endRow;
            eventService.processOnClientCashback(paramsByClient);
            rowNumberProperty.set(currentRow);
        }
        tableCreationTimeProperty.set(table.getAttribute(CREATION_TIME_ATTRIBUTE).get().stringValue());
        rowNumberProperty.remove();
    }
}
