package ru.yandex.direct.jobs.centralizedmonitoring;

import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.tmatesoft.svn.core.SVNException;
import org.tmatesoft.svn.core.SVNURL;
import org.tmatesoft.svn.core.wc.SVNClientManager;
import org.tmatesoft.svn.core.wc.SVNRevision;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcPropertyName;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.annotation.JugglerChecks;
import ru.yandex.direct.juggler.check.model.CheckTag;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.Hourglasses;
import ru.yandex.direct.scheduler.hourglass.HourglassJob;
import ru.yandex.direct.scheduler.support.DirectJob;
import ru.yandex.direct.utils.Condition;

import static java.util.Collections.emptyMap;
import static ru.yandex.direct.common.db.PpcPropertyType.STRING_TO_STRING_MAP;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_0;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_1;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_2;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.JavaKotlinInteropUtils.isKotlinClass;

/**
 * Создание/закрытие тикетов на передачу джоб в централизованный мониторинг
 * <p>
 * Джоба берет список джоб для продакшена.
 * Сравнивает с сохранённым списком джоб в ppcProperties (джоба - тикет)
 * Для каждой джобы проверяет:
 * Если тикета не было, то создаёт тикет в стартреке
 * Если джоба передана в мониторинг, то закрывает тикет
 * Если джоба не передана в мониторинг и тикет закрыт, то переоткрывает тикет
 */

@JugglerCheck(ttl = @JugglerCheck.Duration(hours = 36),
        tags = {DIRECT_PRIORITY_2, CheckTag.DIRECT_PRODUCT_TEAM},
        needCheck = ProductionOnly.class
)
@Hourglass(cronExpression = "0 0 4 * * ?", needSchedule = ProductionOnly.class)
public class JobsCentralizedMonitoringWatchingJob extends DirectJob {

    private static final String LOGIC_PROCESSOR_SUBPATH = "apps/event-sourcing-system/logicprocessor";
    private static final String JOBS_SUBPATH = "jobs";

    private static final String SVN_PATH_TEMPLATE =
            "svn+ssh://arcadia.yandex.ru/arc/trunk/arcadia/direct/%s/src/main/java/%s.%s";

    private static final Logger logger = LoggerFactory.getLogger(JobsCentralizedMonitoringWatchingJob.class);
    private static final PpcPropertyName<Map<String, String>> JOB_TO_TICKET_PROPERTY =
            new PpcPropertyName<>("jobs_centralized_monitoring_watching_job.jobs_tickets", STRING_TO_STRING_MAP);
    private static final Set<CheckTag> MONITORING_TAGS =
            Set.of(DIRECT_PRIORITY_0, DIRECT_PRIORITY_1, DIRECT_PRIORITY_2);

    private final Collection<HourglassJob> allJobs;
    private final SVNClientManager svnClientManager;
    private final ApplicationContext applicationContext;
    private final PpcPropertiesSupport ppcPropertiesSupport;
    private final JobsCentralizedMonitoringWatchingStartrekHelper startrekHelper;

    @Autowired
    public JobsCentralizedMonitoringWatchingJob(Collection<HourglassJob> allJobs,
                                                SVNClientManager svnClientManager,
                                                ApplicationContext applicationContext,
                                                PpcPropertiesSupport ppcPropertiesSupport,
                                                JobsCentralizedMonitoringWatchingStartrekHelper startrekHelper) {
        this.allJobs = allJobs;
        this.svnClientManager = svnClientManager;
        this.applicationContext = applicationContext;
        this.ppcPropertiesSupport = ppcPropertiesSupport;
        this.startrekHelper = startrekHelper;
    }

    @Override
    public void execute() {
        Map<String, String> originalJobToTicketMap = getJobToTicketMap();
        Map<String, String> newJobToTicketMap = new HashMap<>();

        List<HourglassJob> suitableJobs = filterList(allJobs, this::isJobSuitable);
        logger.info("All jobs count: {}, suitable jobs count {}", allJobs.size(), suitableJobs.size());

        for (HourglassJob job : suitableJobs) {
            try {
                handleJob(originalJobToTicketMap, newJobToTicketMap, job);
            } catch (RuntimeException e) {
                logger.error("Exception occurred while handle job {}", job.getClass().getSimpleName(), e);
            }
        }


        if (!newJobToTicketMap.isEmpty()) {
            updatePpcProperty(originalJobToTicketMap, newJobToTicketMap);
        }
    }

    private void handleJob(Map<String, String> originalJobToTicketMap,
                           Map<String, String> newJobToTicketMap,
                           HourglassJob job) {

        String jobName = job.getClass().getSimpleName();
        String ticketId = originalJobToTicketMap.get(jobName);
        if (ticketId == null) {
            ticketId = createNewTicket(job, jobName);
            newJobToTicketMap.put(jobName, ticketId);
        }

        checkExistingTicketStatus(job, ticketId);
    }

    /**
     * Проверяем что джоба:
     * 1) имеет аннотацию Hourglass
     * 2) имеет аннотацию JugglerCheck
     * 3) работает в текущем окружении
     */
    private boolean isJobSuitable(HourglassJob job) {
        Class<? extends HourglassJob> jobClass = job.getClass();

        return (jobClass.isAnnotationPresent(Hourglass.class) || jobClass.isAnnotationPresent(Hourglasses.class))
                && (jobClass.isAnnotationPresent(JugglerCheck.class) || jobClass.isAnnotationPresent(JugglerChecks.class))
                && isNeedSchedule(jobClass);
    }

    /**
     * Проверить, что джоба работтает в текущем окружении
     */
    private boolean isNeedSchedule(Class<? extends HourglassJob> jobClass) {
        Hourglass[] hourglassAnnotations = jobClass.getAnnotationsByType(Hourglass.class);
        for (Hourglass hourglassAnnotation : hourglassAnnotations) {
            Class<? extends Condition> needScheduleClass = hourglassAnnotation.needSchedule();
            try {
                if (applicationContext.getBean(needScheduleClass).evaluate()) {
                    return true;
                }
            } catch (RuntimeException e) {
                logger.error("Exception occurred while checking schedule for Job {}: ", jobClass.getCanonicalName(), e);
            }
        }
        return false;
    }

    /**
     * Обновляем мапу "Имя джобы" -> "Тикет джобы" в ppcProperties
     *
     * @param originalJobToTicketMap - то что лежало в ppcProperties до запуска джобы
     * @param newJobToTicketMap      - новые тикеты
     */
    private void updatePpcProperty(Map<String, String> originalJobToTicketMap, Map<String, String> newJobToTicketMap) {
        logger.info("Update ppcProperty with new values {}", newJobToTicketMap);
        Map<String, String> mergedMap = new HashMap<>(originalJobToTicketMap);
        mergedMap.putAll(newJobToTicketMap);
        ppcPropertiesSupport.get(JOB_TO_TICKET_PROPERTY).set(mergedMap);
    }

    /**
     * Если джоба передана в мониторинг, то тикет должен быть закрыт
     * Если джоба еще не передана в мониторинг, то тикет должен быть открыт
     */
    private void checkExistingTicketStatus(HourglassJob job, String ticketId) {
        boolean isJobPassedToMonitoring = isJobPassedToMonitoring(job.getClass());
        boolean isTicketOpen = startrekHelper.isTicketOpen(ticketId);

        if (isJobPassedToMonitoring) {
            if (isTicketOpen) {
                startrekHelper.closeTicket(ticketId);
            }
        } else if (!isTicketOpen) {
            logger.warn("Job is not passed to monitoring, but ticket {} is closed", ticketId);
            startrekHelper.reopenTicket(ticketId);
        }
    }

    private String createNewTicket(HourglassJob job, String jobName) {
        String author = getAuthor(job.getClass());
        return startrekHelper.createTicket(jobName, author);
    }

    /**
     * Если у джобы есть тег priority_N, значит она передана в мониторинг
     */
    private boolean isJobPassedToMonitoring(Class<? extends HourglassJob> jobClass) {
        JugglerCheck[] jugglerCheckAnnotations = jobClass.getAnnotationsByType(JugglerCheck.class);

        return Arrays.stream(jugglerCheckAnnotations)
                .map(JugglerCheck::tags)
                .flatMap(Arrays::stream)
                .anyMatch(MONITORING_TAGS::contains);
    }

    /**
     * Определяем автора класса по первой строчке из svn
     */
    private String getAuthor(Class<? extends HourglassJob> jobClass) {
        String classCanonicalPath = jobClass.getCanonicalName().replace(".", "/");
        String subPath = classCanonicalPath.contains("/logicprocessor/") ? LOGIC_PROCESSOR_SUBPATH : JOBS_SUBPATH;
        String classExtension = isKotlinClass(jobClass) ? "kt" : "java";

        String fullPath = String.format(SVN_PATH_TEMPLATE, subPath, classCanonicalPath, classExtension);

        try {
            return getAuthorFromSvn(fullPath);
        } catch (SVNException e) {
            logger.error("Class could not be found {} in arcadia", jobClass);
            throw new RuntimeException(e);
        }
    }

    private String getAuthorFromSvn(String fullSvnPath) throws SVNException {
        SVNURL svnFilePath = SVNURL.parseURIEncoded(fullSvnPath);
        AuthorByFirstLineHandler handler = new AuthorByFirstLineHandler();

        svnClientManager.getLogClient()
                .doAnnotate(svnFilePath, SVNRevision.HEAD, SVNRevision.create(0), SVNRevision.HEAD, handler);

        return handler.getFirstLineAuthor();

    }

    /**
     * Достать мапу "Имя джобы" -> "Тикет джобы"
     */
    private Map<String, String> getJobToTicketMap() {
        return ppcPropertiesSupport.get(JOB_TO_TICKET_PROPERTY)
                .getOrDefault(emptyMap());
    }


}
