package ru.yandex.direct.jobs.centralizedmonitoring;

import java.util.List;
import java.util.Optional;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import ru.yandex.direct.env.Environment;
import ru.yandex.startrek.client.Session;
import ru.yandex.startrek.client.model.CommentCreate;
import ru.yandex.startrek.client.model.Issue;
import ru.yandex.startrek.client.model.IssueCreate;
import ru.yandex.startrek.client.model.IssueUpdate;
import ru.yandex.startrek.client.model.StatusRef;
import ru.yandex.startrek.client.model.User;

import static java.util.stream.Collectors.toList;

@Service
public class JobsCentralizedMonitoringWatchingStartrekHelper {

    private static final Logger logger = LoggerFactory.getLogger(JobsCentralizedMonitoringWatchingStartrekHelper.class);
    private static final String QUEUE = "DIRECT";
    private static final String TAG_NAME = "jobs_to_support_monitoring";
    private static final String SUMMARY_TEMPLATE = "Передача в поддержку: %s";
    private static final String FIXED_STATUS = "fixed";
    private static final String CLOSE_TRANSITION = "close";
    private static final String REOPEN_TRANSITION = "reopen";
    private static final String OPEN_STATUS_KEY = "open";
    private static final String DESCRIPTION =
            "((https://docs.yandex-team.ru/direct-dev/concepts/dev/jobs-monitoring " +
                    "Документация по мониторингу джоб))\n" +
                    "После передачи джобы в поддержку тикет закроется автоматически.\n\n" +
                    "//Тикет создан с помощью " + JobsCentralizedMonitoringWatchingJob.class.getSimpleName() + "//";


    private final Session session;

    public JobsCentralizedMonitoringWatchingStartrekHelper(Session session) {
        this.session = session;
    }

    public String createTicket(String jobName, String author) {
        if (Environment.getCached().isDevelopment()) {
            author = "ppc";
        }

        // проверяем что пользователь существует. Иначе user = null, чтобы IssueCreate не упал
        User user = session.users().getO(author).getOrNull();

        if (user == null) {
            logger.warn("User {} not found", author);
        }

        // На всякий случай ищем похожие тикеты и пишем в лог
        findTicket(jobName).ifPresent(existingTicketId ->
                logger.warn("Similar ticket for job {} already exists: {}", jobName, existingTicketId));

        IssueCreate issueCreate = IssueCreate.builder()
                .queue(QUEUE)
                .type("task")
                .summary(getSummary(jobName))
                .description(DESCRIPTION)
                .tags(TAG_NAME)
                .assignee(user)
                .build();

        Issue issue = session.issues().create(issueCreate);
        logger.info("Ticket created: {} for job {}", issue.getKey(), jobName);
        return issue.getKey();
    }

    /**
     * Найти тикет по jobName
     */
    private Optional<String> findTicket(String jobName) {
        String query =
                String.format("\"Queue\": \"%s\" " +
                                "AND \"Summary\": \"%s\" " +
                                "AND \"Tags\": \"%s\"",
                        QUEUE, getSummary(jobName), TAG_NAME);

        String expectedSummary = getSummary(jobName);

        // поиск в стартреке нечеткий, поэтому дополнительно ищем по точному совпадению
        List<Issue> issueList = session.issues().find(query).stream()
                .filter(issue -> expectedSummary.equals(issue.getSummary()))
                .collect(toList());

        return issueList.isEmpty() ? Optional.empty() : Optional.of(issueList.get(0).getKey());
    }

    private String getSummary(String jobName) {
        return String.format(SUMMARY_TEMPLATE, jobName);
    }

    public boolean isTicketOpen(String ticketId) {
        StatusRef status = session.issues().get(ticketId).getStatus();
        return OPEN_STATUS_KEY.equalsIgnoreCase(status.getKey());
    }

    public void closeTicket(String ticketKey) {
        IssueUpdate issueUpdate = IssueUpdate.builder()
                .comment(CommentCreate.comment("Джоба передана в мониторинг").build())
                .resolution(FIXED_STATUS)
                .build();

        session.issues().update(ticketKey, issueUpdate);
        session.transitions().execute(ticketKey, CLOSE_TRANSITION);
        logger.info("Ticket closed {}", ticketKey);
    }

    public void reopenTicket(String ticketKey) {
        session.transitions().execute(ticketKey, REOPEN_TRANSITION);
        logger.info("Ticket reopened {}", ticketKey);
    }

}
