package ru.yandex.direct.jobs.clickhousecleaner;

import java.io.IOException;
import java.net.URI;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.time.Duration;
import java.time.Instant;
import java.time.Period;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.collect.Sets;
import com.typesafe.config.ConfigFactory;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;

import ru.yandex.clickhouse.ClickHouseDataSource;
import ru.yandex.direct.clickhouse.ClickHouseUtils;
import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.db.config.DbConfig;
import ru.yandex.direct.db.config.DbConfigFactory;
import ru.yandex.direct.dbutil.wrapper.SimpleDb;
import ru.yandex.direct.env.NonDevelopmentEnvironment;
import ru.yandex.direct.env.TypicalEnvironment;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectJob;
import ru.yandex.direct.utils.InterruptedRuntimeException;
import ru.yandex.direct.utils.JsonUtils;
import ru.yandex.misc.io.ClassPathResourceInputStreamSource;

import static java.util.Arrays.asList;
import static java.util.Collections.singletonList;
import static java.util.stream.Collectors.toMap;
import static ru.yandex.direct.common.db.PpcPropertyNames.PPCHOUSE_CLEANER_REAL_REMOVE;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_2;

/**
 * Удаляет старые партиции табличек в ppchouse-cloud,
 * время жизни которых указано в table-store-periods.conf
 */
@Hourglass(periodInSeconds = 3600, needSchedule = TypicalEnvironment.class)
@ParametersAreNonnullByDefault
@JugglerCheck(ttl = @JugglerCheck.Duration(days = 3), needCheck = NonDevelopmentEnvironment.class, tags = {DIRECT_PRIORITY_2})
public class ClickHouseCleanerJob extends DirectJob {
    private static final Logger logger = LoggerFactory.getLogger(ClickHouseCleanerJob.class);

    private static final Duration GET_HOSTS_TIMEOUT = Duration.ofSeconds(10);
    private static final Duration REAL_REMOVE_PROPERTY_EXPIRATION = Duration.ofSeconds(10);

    static final long MAX_DB_TIME_DIFF_SEC = 3600;

    @VisibleForTesting
    final DbConfig clickhouseConfig;

    private final PpcProperty<Boolean> ppchouseCleanerRealRemove;
    private final String readHostsUrl;
    private String tableStoreConfigPath;

    private final List<String> initialClickhouseConfigHosts;

    @Autowired
    public ClickHouseCleanerJob(
            PpcPropertiesSupport ppcPropertiesSupport,
            DbConfigFactory dbConfigFactory,
            @Value("${clickhouse.read_hosts_url:#{null}}") String readHostsUrl,
            @Value("${clickhouse_cleaner.table_store_config_path}") String tableStoreConfigPath
    ) {
        ppchouseCleanerRealRemove =
                ppcPropertiesSupport.get(PPCHOUSE_CLEANER_REAL_REMOVE, REAL_REMOVE_PROPERTY_EXPIRATION);

        clickhouseConfig = dbConfigFactory.get(SimpleDb.CLICKHOUSE_CLOUD_WRITER.toString());
        initialClickhouseConfigHosts = clickhouseConfig.getHosts();

        this.tableStoreConfigPath = tableStoreConfigPath;
        this.readHostsUrl = readHostsUrl;
    }

    @Override
    public void execute() {
        var hosts = getHosts();
        if (hosts.isEmpty()) {
            throw new IllegalStateException("List of clickhouse hosts to clear is empty.");
        }

        var tableStorePeriods = getTableStorePeriods();

        var failedHosts = new ArrayList<String>();

        int i = 0;
        for (String host : hosts) {
            logger.info("connecting to host {}:{} ({}/{})", host, clickhouseConfig.getPort(), ++i, hosts.size());
            try (Connection conn = getClickHouseConnection(host)) {
                // сверяем системное время и список табличек

                var nowInstant = Instant.now();
                checkHostTime(conn, nowInstant);
                checkUnknownTables(conn, tableStorePeriods.keySet());

                int failedPartitionsCount = clearTables(conn, tableStorePeriods, host, nowInstant,
                        ppchouseCleanerRealRemove.getOrDefault(false));

                if (failedPartitionsCount > 0) {
                    throw new IllegalStateException("failed to remove " + failedPartitionsCount +
                            " partitions from the host " + host);
                }

            } catch (SQLException | RuntimeException e) {
                // by the way clickhouse driver throws RuntimeException instead of SQLTimeoutException
                logger.error("exception for host " + host + ":" + clickhouseConfig.getPort(), e);
                failedHosts.add(host);
                // continue other hosts
            }
        }

        if (!failedHosts.isEmpty()) {
            throw new IllegalStateException(failedHosts.size() + " hosts have failed with exception: " +
                    StreamEx.of(failedHosts).joining(", "));
        }
    }

    List<String> getHosts() {
        if (readHostsUrl == null) {
            logger.info("got read_hosts_url parameter null, taking hosts from clickhouse config: {}",
                    initialClickhouseConfigHosts);
            return initialClickhouseConfigHosts;
        }

        HttpRequest request = HttpRequest.newBuilder()
                .timeout(GET_HOSTS_TIMEOUT)
                .uri(URI.create(readHostsUrl))
                .build();
        try {
            HttpResponse<String> response = HttpClient.newHttpClient().send(request,
                    HttpResponse.BodyHandlers.ofString());

            return asList(JsonUtils.MAPPER.readValue(response.body(), String[].class));

        } catch (RuntimeException | IOException e) {
            throw new IllegalStateException("Can not get hosts from " + readHostsUrl, e);
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new InterruptedRuntimeException(e);
        }
    }

    Connection getClickHouseConnection(String host) throws SQLException {
        clickhouseConfig.setHosts(singletonList(host));

        return new ClickHouseDataSource(clickhouseConfig.getJdbcUrl()).getConnection(
                clickhouseConfig.getUser(), clickhouseConfig.getPass());
    }

    static void checkHostTime(Connection conn, Instant serverInstant) {
        long serverTimestamp = serverInstant.getEpochSecond();
        try (ResultSet timestampResult = conn.prepareStatement("select toUnixTimestamp(now())").executeQuery()) {
            timestampResult.next();
            long dbTimestamp = timestampResult.getLong(1);

            Preconditions.checkState(Math.abs(serverTimestamp - dbTimestamp) < MAX_DB_TIME_DIFF_SEC,
                    "Too large time difference between server utc timestamp %s and db host timestamp %s",
                    serverTimestamp, dbTimestamp);
        } catch (SQLException e) {
            throw new IllegalStateException("Sql exception in checkHostTime", e);
        }
    }

    static void checkUnknownTables(Connection conn, Set<String> tableNamesToClear) {
        try (ResultSet tablesResult = conn.prepareStatement("show tables").executeQuery()) {

            Set<String> tableNames = new HashSet<>();
            while (tablesResult.next()) {
                tableNames.add(tablesResult.getString(1));
            }
            if (!tableNames.containsAll(tableNamesToClear)) {
                logger.warn("Settings for clearing contain unknown tables: {}",
                        Sets.difference(tableNamesToClear, tableNames));
            }
        } catch (SQLException e) {
            throw new IllegalStateException("Sql exception in checkUnknownTables", e);
        }
    }

    /**
     * @return количество ошибок удаления партиций. 0 если всё в порядке.
     */
    int clearTables(Connection conn, Map<String, Period> tableMonthStoreCounts, String host,
                    Instant nowInstant, boolean realRemove) {
        // потому что джава не умеет instant.minus(месяцы)
        ZonedDateTime utcNowDateTime = ZonedDateTime.ofInstant(nowInstant, ZoneOffset.UTC);

        int failedPartitionsCount = 0;
        for (Map.Entry<String, Period> entry : tableMonthStoreCounts.entrySet()) {
            var table = entry.getKey();
            var period = entry.getValue();
            if (period.isNegative() || period.isZero()) { // оставлять навсегда
                continue;
            }

            var sql = getSqlToFindPartitionIds(table, period, utcNowDateTime);

            try (ResultSet rs = conn.prepareStatement(sql).executeQuery()) {
                while (rs.next()) {
                    var partitionId = rs.getString(1);
                    var maxDate = rs.getString(2);
                    logger.info("for host {} table {} delete partition_id {} older period {}, last record was at {}, " +
                                    "realRemove={}",
                            host, table, partitionId, period, maxDate, realRemove);

                    if (realRemove) {
                        boolean succeeded = deletePartition(conn, partitionId, table, host);
                        failedPartitionsCount += succeeded ? 0 : 1;
                    }
                }
            } catch (SQLException e) {
                throw new IllegalStateException("Sql exception in clearTables", e);
            }
        }
        return failedPartitionsCount;
    }

    /**
     * @return было ли удаление успешным
     */
    boolean deletePartition(Connection conn, String partitionId, String table, String host) {
        try (Statement statement = conn.createStatement()) {
            statement.execute("alter table " + ClickHouseUtils.quoteName(table) +
                    " drop partition id " + ClickHouseUtils.quote(partitionId));

            logger.info("Successfully removed partition_id {} from table {}, host {}", partitionId, table, host);
            return true;
        } catch (SQLException e) {
            logger.error("Can not drop partition_id " + partitionId +
                    " from table " + table + " on host " + host, e);
            return false;
        }
    }

    /**
     * @return Map: имя таблички -> период хранения. Значения 0 и меньше значат хранить всегда.
     */
    @Nonnull
    Map<String, Period> getTableStorePeriods() {
        return readStringPeriodMap(this.tableStoreConfigPath);
    }

    @Nonnull
    @VisibleForTesting
    Map<String, Period> readStringPeriodMap(String path) {
        try {
            String configStr = new ClassPathResourceInputStreamSource(path)
                    .readText();
            var storeConfig = ConfigFactory.parseString(configStr)
                    .getConfig("table-store-count");

            return storeConfig.entrySet().stream()
                    .collect(toMap(e -> e.getKey(), e -> Period.parse(storeConfig.getString(e.getKey()))));

        } catch (RuntimeException e) {
            throw new IllegalStateException("Can not load config from " + path, e);
        }
    }

    /**
     * Вычисляет все partition_id, у которых во всех parts max_time старше (now - period).
     * В ответе будут колонки: partition_id, время последней записи
     */
    private static String getSqlToFindPartitionIds(String table, Period period, ZonedDateTime utcNowDateTime) {
        // вычисляем дату до которой удалять
        long deleteTimestamp = utcNowDateTime.minus(period).toEpochSecond();

        return "SELECT\n" +
                "    partition_id,\n" +
                "    max(max_date) \n" +
                "FROM system.parts\n" +
                "WHERE table = " + ClickHouseUtils.quote(table) + "\n" +
                "AND max_date != '0000-00-00'\n" +
                "GROUP BY partition_id\n" +
                "HAVING max(max_date) < toDate(" + deleteTimestamp + ")";
    }
}
