package ru.yandex.direct.jobs.clientavatars.gc;

import java.util.ArrayList;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Service;

import ru.yandex.direct.avatars.client.model.AvatarId;
import ru.yandex.direct.core.entity.freelancer.model.ClientAvatar;
import ru.yandex.direct.core.entity.freelancer.model.ClientAvatarsHost;
import ru.yandex.direct.core.entity.freelancer.repository.ClientAvatarRepository;
import ru.yandex.direct.core.entity.freelancer.service.AvatarsClientPool;
import ru.yandex.direct.core.entity.freelancer.service.AvatarsConfigNameConverter;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;

import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Основная часть джобы, подчищающей аватарки.
 * Находит неиспользуемые аватарки (только для своей среды - т.е. в тестинге удаляет только аватарки тестовой
 * аватарницы), помечает их удалёнными в реестре, удаляет их из аватарницы, после этого удаляет их из реестра.
 * Если не удалось удалить аватарку в аватарнице, то попытка будет повторена на следующем круге.
 */
@Service
public class ClientAvatarsDeleteService {
    // по истечении суток непривязанная к карточке аватарка считается неиспользуемой.
    private static final int AVATARS_EXPIRATION_PERIOD = 24 * 60 * 60;
    private static final Logger logger = LoggerFactory.getLogger(ClientAvatarsDeleteService.class);

    private final ClientAvatarRepository clientAvatarRepository;
    private final AvatarsClientPool avatarsClientPool;
    private final ClientAvatarsHost clientAvatarsHost;

    @Autowired
    public ClientAvatarsDeleteService(
            ClientAvatarRepository clientAvatarRepository,
            @Qualifier("freelancersAvatarsClientPool") AvatarsClientPool avatarsClientPool,
            AvatarsConfigNameConverter avatarsConfigNameConverter) {
        this.clientAvatarRepository = clientAvatarRepository;
        this.avatarsClientPool = avatarsClientPool;
        clientAvatarsHost = avatarsConfigNameConverter.getHost(avatarsClientPool.getDefaultConfigName());
    }

    int doWork(int shard) {
        return deleteUnusedClientAvatars(shard);
    }

    private int deleteUnusedClientAvatars(int shard) {
        List<ClientAvatar> unusedAvatars = clientAvatarRepository.getUnused(
                shard, clientAvatarsHost, AVATARS_EXPIRATION_PERIOD);
        markAvatarsDeleted(shard, unusedAvatars);
        List<Long> removedClientAvatarIds = deleteAvatarsFromMdsAvatars(unusedAvatars);
        return clientAvatarRepository.delete(shard, removedClientAvatarIds);
    }

    private void markAvatarsDeleted(int shard, List<ClientAvatar> unusedAvatars) {
        List<AppliedChanges<ClientAvatar>> appliedChanges = mapList(unusedAvatars, avatar -> {
            ModelChanges<ClientAvatar> modelChanges = new ModelChanges<>(avatar.getId(), ClientAvatar.class);
            modelChanges.process(true, ClientAvatar.IS_DELETED);
            return modelChanges.applyTo(avatar);
        });
        clientAvatarRepository.update(shard, appliedChanges);
    }

    private List<Long> deleteAvatarsFromMdsAvatars(List<ClientAvatar> unusedAvatars) {
        List<Long> removedClientAvatarIds = new ArrayList<>();
        for (ClientAvatar clientAvatar : unusedAvatars) {
            AvatarId avatarId;
            try {
                avatarId = AvatarId.fromIdString(clientAvatar.getExternalId());
            } catch (IllegalArgumentException e) {
                logger.error("Avatar with wrong external id was found: id={}, clientId={}, externalId={}.",
                        clientAvatar.getId(), clientAvatar.getClientId(), clientAvatar.getExternalId());
                removedClientAvatarIds.add(clientAvatar.getId()); // запись бесполезна без корректного ExternalId.
                continue;
            }
            if (avatarsClientPool.getDefaultClient().delete(avatarId)) {
                // даже если в этом месте выполнение почему-то вдруг прервётся и информация о только что удалённой
                // аватарке потеряется - это не критично, AvatarsClient#delete возвращает true для уже удалёной
                // аватарки, поэтому, запись из базы будет удалена при следующем вызове deleteUnusedClientAvatars()
                logger.info("Successfully delete avatar with id={}, clientId={} and externalId={} from MDS Avatars.",
                        clientAvatar.getId(), clientAvatar.getClientId(), clientAvatar.getExternalId());
                removedClientAvatarIds.add(clientAvatar.getId());
            }
        }
        return removedClientAvatarIds;
    }
}
