package ru.yandex.direct.jobs.clientmeasurersettings;

import java.time.Instant;
import java.util.ArrayList;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Strings;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod;
import ru.yandex.direct.core.entity.client.model.ClientMeasurerSystem;
import ru.yandex.direct.core.entity.client.model.MediascopeClientMeasurerSettings;
import ru.yandex.direct.core.entity.client.service.ClientMeasurerSettingsService;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.env.TypicalEnvironment;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification;
import ru.yandex.direct.juggler.check.model.CheckTag;
import ru.yandex.direct.juggler.check.model.NotificationRecipient;
import ru.yandex.direct.mediascope.MediascopeClient;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectShardedJob;
import ru.yandex.direct.tracing.Trace;

import static java.lang.Math.min;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_0;
import static ru.yandex.direct.utils.JsonUtils.fromJson;
import static ru.yandex.direct.utils.JsonUtils.toJson;


/**
 * Джоба обновления токенов для интеграции с Mediascope
 * Джоба нужна для поддержания актуальности токенов авторизации в ЛК Mediascope:
 * 1. Собираем все настройки Mediascope из client_measurer_settings
 * 2. Находим токены, которые протухнут через {@value #EXPIRATION_PERIOD_IN_SECONDS} секунд
 * 3. Запрашиваем новые токены через API Mediascope
 * 4. Обновляем настройки в базе
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(hours = 2, minutes = 5),
        tags = {DIRECT_PRIORITY_0, CheckTag.DIRECT_PRODUCT_TEAM},
        needCheck = ProductionOnly.class,
        notifications = @OnChangeNotification(
                recipient = NotificationRecipient.LOGIN_AMMSAID,
                method = NotificationMethod.TELEGRAM,
                status = {JugglerStatus.OK, JugglerStatus.CRIT}
        )
)
@Hourglass(periodInSeconds = 60 * 60, needSchedule = TypicalEnvironment.class)
@ParametersAreNonnullByDefault
public class MediascopeSettingsRefresherJob extends DirectShardedJob {
    private static final Logger logger = LoggerFactory.getLogger(MediascopeSettingsRefresherJob.class);
    private static final int EXPIRATION_PERIOD_IN_SECONDS = 180 * 60;

    private final MediascopeClient mediascopeClient;
    private final ClientMeasurerSettingsService clientMeasurerSettingsService;

    @Autowired
    public MediascopeSettingsRefresherJob(
            MediascopeClient mediascopeClient,
            ClientMeasurerSettingsService clientMeasurerSettingsService
    ) {
        this.mediascopeClient = mediascopeClient;
        this.clientMeasurerSettingsService = clientMeasurerSettingsService;
    }

    /**
     * Конструктор для тестов.
     */
    public MediascopeSettingsRefresherJob(
            int shardId,
            MediascopeClient mediascopeClient,
            ClientMeasurerSettingsService clientMeasurerSettingsService
    ) {
        super(shardId);
        this.mediascopeClient = mediascopeClient;
        this.clientMeasurerSettingsService = clientMeasurerSettingsService;
    }

    @Override
    public void execute() {
        var clientMeasurerSettingsList =
                clientMeasurerSettingsService.getByMeasurerSystem(this.getShard(), ClientMeasurerSystem.MEDIASCOPE);

        List<Long> badTokenClientIds = new ArrayList<>();

        try (var ignore = Trace.current().profile("mediascope_integration:tokens_job", "update_tokens_for_shard")) {
            for (var clientMeasurerSettings : clientMeasurerSettingsList) {
                try {
                    var settings = fromJson(
                            clientMeasurerSettings.getSettings(), MediascopeClientMeasurerSettings.class);

                    // refresh tokens, if EXPIRATION_PERIOD_IN_SECONDS or less left until expiration
                    long expirationThreshold = getEpochSecondFromNow(EXPIRATION_PERIOD_IN_SECONDS);
                    if (settings.getExpiresAt() < expirationThreshold) {
                        var tokensResponse = mediascopeClient.getFreshTokens(settings.getRefreshToken());
                        if (tokensResponse != null) {
                            settings.withAccessToken(tokensResponse.getAccessToken())
                                    .withRefreshToken(tokensResponse.getRefreshToken())
                                    .withExpiresAt(tokensResponse.getExpiresIn() + Instant.now().getEpochSecond());
                        } else {
                            logger.error(
                                    "Error refreshing mediascope token for clientId {}",
                                    clientMeasurerSettings.getClientId());
                            badTokenClientIds.add(clientMeasurerSettings.getClientId());
                        }
                    }

                    // prefix might be changed manually, so we try to refresh it once per hour
                    var prefixResponse = mediascopeClient.getPrefix(settings.getAccessToken());
                    if (prefixResponse != null && !Strings.isNullOrEmpty(prefixResponse.getPrefix())) {
                        settings.withTmsecprefix(prefixResponse.getPrefix());
                    } else {
                        logger.error(
                                "Error getting mediascope prefix (tmsecprefix) for clientId {}",
                                clientMeasurerSettings.getClientId());
                    }

                    // username shouldn't ever change, so we refresh it only if it was not filled initially
                    if (Strings.isNullOrEmpty(settings.getUsername())) {
                        var userInfoResponse = mediascopeClient.getUserInfo(settings.getAccessToken());
                        if (userInfoResponse != null && !Strings.isNullOrEmpty(userInfoResponse.getUsername())) {
                            settings.withUsername(userInfoResponse.getUsername());
                        } else {
                            logger.error(
                                    "Error getting mediascope username for clientId {}",
                                    clientMeasurerSettings.getClientId());
                        }
                    }

                    clientMeasurerSettings.setSettings(toJson(settings));
                    clientMeasurerSettingsService.update(this.getShard(), clientMeasurerSettings);
                } catch (Exception e) {
                    logger.error(String.format(
                            "Error refreshing mediascope token for clientId %s",
                            clientMeasurerSettings.getClientId()), e);
                    badTokenClientIds.add(clientMeasurerSettings.getClientId());
                }
            }
        }

        if (!badTokenClientIds.isEmpty()) {
            List<Long> badTokenIdsForMsg = badTokenClientIds.subList(0, min(badTokenClientIds.size(), 20));
            setJugglerStatus(JugglerStatus.CRIT,
                    String.format("Couldn't refresh %d tokens for clients: ", badTokenClientIds.size()) + badTokenIdsForMsg);
        } else {
            setJugglerStatus(JugglerStatus.OK, "All clients refreshed tokens");
        }
    }

    private long getEpochSecondFromNow(int period) {
        return Instant.now().plusSeconds(period).getEpochSecond();
    }
}
