package ru.yandex.direct.jobs.clientsdailyspent;

import java.time.Duration;
import java.util.Objects;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.CampaignTypeKinds;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsType;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.jobs.configuration.DirectExportYtClustersParametersSource;
import ru.yandex.direct.jobs.util.yql.CommonYqlExport;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.model.CheckTag;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectParameterizedJob;
import ru.yandex.direct.scheduler.support.ParameterizedBy;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YtCluster;

import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_1_NOT_READY;

/**
 * Расчёт данных в YT о тратах рекламодателей (за весь период и последние 28 дней показов) и остатках.
 * Производится в разрезе: клиент / бренд / агентство. Заменяет собой перловый
 * {@code Direct::YT::Export::ClientsMonthSpentEstimate}.
 *
 * @see ImportEstimateFromYtJob
 */
@Hourglass(periodInSeconds = 3600, needSchedule = ProductionOnly.class)
@JugglerCheck(ttl = @JugglerCheck.Duration(hours = 7),

        //PRIORITY: уточнить у gerdler терпим ли мы не посчитанные траты для баллов неделю, или это все таки 2
        tags = {DIRECT_PRIORITY_1_NOT_READY, CheckTag.YT, CheckTag.GROUP_INTERNAL_SYSTEMS},
        needCheck = ProductionOnly.class)
@ParameterizedBy(parametersSource = DirectExportYtClustersParametersSource.class)
@ParametersAreNonnullByDefault
public class CalcEstimateJob extends DirectParameterizedJob<YtCluster> {
    private static final Logger logger = LoggerFactory.getLogger(CalcEstimateJob.class);

    private static final String QUERY_RESOURCE = "classpath:///clientsdailyspent/ClientsDailySpentEstimate.yql";
    static final String EXPORT_PATH = "import/clients_daily_spent_estimate";

    /**
     * Интервал пересчета данных.
     * Старая выгрузка (по direct/db) срабатывала раз в день, конечные данные тоже обрабатываются раз в сутки.
     * Текущий расчет длится около часа, поэтому обновляем данные примерно раз в день
     */
    private static final Duration GENERATION_INTERVAL = Duration.ofHours(22);

    private final CommonYqlExport.Builder exportBuilder;
    private final DirectExportYtClustersParametersSource parametersSource;

    @Autowired
    public CalcEstimateJob(DirectExportYtClustersParametersSource parametersSource, YtProvider ytProvider) {
        this.parametersSource = parametersSource;

        exportBuilder = new CommonYqlExport.Builder(logger, ytProvider, QUERY_RESOURCE, EXPORT_PATH)
                .withBindingsProvider(CalcEstimateJob::getBindings);
    }

    @Override
    public void execute() {
        YtCluster ytCluster = parametersSource.convertStringToParam(getParam());
        exportBuilder.build(ytCluster).generateIfNeeded(GENERATION_INTERVAL);
    }

    private static Object[] getBindings(CommonYqlExport.Context ctx) {
        String types = StreamEx.of(CampaignTypeKinds.API5_CAMP_SUM_REST_FOR_UNITS)
                .map(CampaignType::toSource)
                .filter(Objects::nonNull)
                .map(CampaignsType::getLiteral)
                .joining(",");

        return new Object[]{
                ctx.getDestinationTablePathAsString(),
                types
        };
    }
}
