package ru.yandex.direct.jobs.clientsdailyspent;

import java.math.BigDecimal;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod;
import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.common.util.RelaxedWorker;
import ru.yandex.direct.core.entity.client.model.AggregationType;
import ru.yandex.direct.core.entity.client.model.ClientSpent;
import ru.yandex.direct.core.entity.client.repository.ClientSpentRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;
import ru.yandex.direct.env.NonDevelopmentEnvironment;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.env.TestingOnly;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification;
import ru.yandex.direct.juggler.check.model.CheckTag;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectJob;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.direct.ytwrapper.model.YtOperator;
import ru.yandex.direct.ytwrapper.model.YtTable;

import static ru.yandex.direct.common.db.PpcPropertyNames.CLIENTS_DAILY_SPENT_LAST_UPLOAD_TIME;
import static ru.yandex.direct.jobs.clientsdailyspent.CalcEstimateJob.EXPORT_PATH;
import static ru.yandex.direct.jobs.configuration.JobsEssentialConfiguration.DEFAULT_YT_CLUSTER;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_2;
import static ru.yandex.direct.juggler.check.model.NotificationRecipient.CHAT_INTERNAL_SYSTEMS_MONITORING;
import static ru.yandex.direct.ytwrapper.YtPathUtil.generatePath;

/**
 * Импорт из YT данных о тратах рекламодателей (за весь период и последние 28 дней показов) и остатках,
 * посчитанных {@link CalcEstimateJob}.
 * Результат используется затем для расчета дневного лимита баллов API5 скриптом {@code api5CalcUnitsLimit.pl}.
 */
@Hourglass(periodInSeconds = 1800, needSchedule = NonDevelopmentEnvironment.class)
@JugglerCheck(ttl = @JugglerCheck.Duration(hours = 8),
        tags = {DIRECT_PRIORITY_2, CheckTag.YT, CheckTag.GROUP_INTERNAL_SYSTEMS},
        notifications = {
                @OnChangeNotification(recipient = CHAT_INTERNAL_SYSTEMS_MONITORING,
                        status = {JugglerStatus.OK, JugglerStatus.CRIT},
                        method = NotificationMethod.TELEGRAM),
        },
        needCheck = ProductionOnly.class)
@JugglerCheck(ttl = @JugglerCheck.Duration(hours = 5),
        tags = {DIRECT_PRIORITY_2, CheckTag.YT, CheckTag.GROUP_INTERNAL_SYSTEMS, CheckTag.JOBS_RELEASE_REGRESSION},
        needCheck = TestingOnly.class)
@ParametersAreNonnullByDefault
public class ImportEstimateFromYtJob extends DirectJob {
    private static final Logger logger = LoggerFactory.getLogger(ImportEstimateFromYtJob.class);

    private static final int YT_READ_CHUNK_SIZE = 10_000;
    private static final int DB_WRITE_CHUNK_SIZE = 1_000;
    private static final RelaxedWorker RELAXED_WORKER = new RelaxedWorker(2.0);
    // для отладки можно поменять префикс на свой, главное случайно не закоммитить
    private static final String PATH_PREFIX = "//home/direct";
    private static final YtTable TABLE = new YtTable(generatePath(PATH_PREFIX, EXPORT_PATH));

    private final ShardHelper shardHelper;
    private final ClientSpentRepository clientSpentRepository;
    private final YtProvider ytProvider;
    private final YtCluster ytCluster;
    private final PpcProperty<String> lastUploadTimeProp;

    @Autowired
    public ImportEstimateFromYtJob(ShardHelper shardHelper, ClientSpentRepository clientSpentRepository,
                                   YtProvider ytProvider, @Qualifier(DEFAULT_YT_CLUSTER) YtCluster ytCluster,
                                   PpcPropertiesSupport ppcPropertiesSupport) {
        this.shardHelper = shardHelper;
        this.clientSpentRepository = clientSpentRepository;
        this.ytProvider = ytProvider;
        this.ytCluster = ytCluster;

        this.lastUploadTimeProp = ppcPropertiesSupport.get(CLIENTS_DAILY_SPENT_LAST_UPLOAD_TIME);
    }

    @Override
    public void execute() {
        YtOperator ytOperator = ytProvider.getOperator(ytCluster);

        if (!ytOperator.exists(TABLE)) {
            logger.error("Table for import {} doesn't exists", TABLE);
            setJugglerStatus(JugglerStatus.CRIT, "Table doesn't exists");
            return;
        }

        String uploadTime = ytOperator.readTableUploadTime(TABLE);
        String lastUploadTime = lastUploadTimeProp.get();
        if (lastUploadTime == null || uploadTime.compareTo(lastUploadTime) <= 0) {
            logger.info("Table data already imported");
            return;
        }

        logger.debug("import data from table uploaded at {}", uploadTime);
        long count = ytOperator.readTableSnapshot(TABLE, new ClientsDailySpentEstimateTableRow(), this::ytRowMapper,
                this::saveDataChunk, YT_READ_CHUNK_SIZE);
        lastUploadTimeProp.set(uploadTime);
        logger.info("Successfully imported {} rows from table", count);
    }

    private void saveDataChunk(List<ClientSpent> clientsSpending) {
        shardHelper.groupByShard(clientsSpending, ShardKey.CLIENT_ID, ImportEstimateFromYtJob::clientIdExtractor)
                .chunkedBy(DB_WRITE_CHUNK_SIZE)
                .forEach(this::saveDataChunkInShard);
    }

    private void saveDataChunkInShard(int shard, List<ClientSpent> clientsSpending) {
        if (shard < 1) {
            logger.warn("Skip saving: unknown shard {} for data {}", shard, clientsSpending);
            setJugglerStatus(JugglerStatus.WARN, "Got data with unknown shard, check log for details");
            return;
        }
        RELAXED_WORKER.runAndRelax(() -> clientSpentRepository.replaceClientsSpending(shard, clientsSpending));
    }

    private static long clientIdExtractor(ClientSpent spent) {
        return spent.getClientId().asLong();
    }

    private ClientSpent ytRowMapper(ClientsDailySpentEstimateTableRow row) {
        return new ClientSpent()
                .withClientId(ClientId.fromLong(row.getClientId()))
                .withType(AggregationType.valueOf(row.getType().toUpperCase()))
                .withActive28DaysSum(doubleToDecimal(row.getActive28daysSumRub()))
                .withDailySpentEstimate(doubleToDecimal(row.getDailySpentRubEstimate()))
                .withRestSum(doubleToDecimal(row.getSumRestRub()))
                .withTotalSum(doubleToDecimal(row.getTotalSumRub()));
    }

    private static BigDecimal doubleToDecimal(String value) {
        return new BigDecimal(value);
    }
}
