package ru.yandex.direct.jobs.communication;

import java.time.Instant;
import java.time.LocalDateTime;
import java.time.OffsetDateTime;
import java.time.ZoneOffset;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.apache.commons.lang3.tuple.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod;
import ru.yandex.direct.core.entity.communication.model.CommunicationEvent;
import ru.yandex.direct.core.entity.communication.model.CommunicationEventType;
import ru.yandex.direct.core.entity.communication.model.CommunicationEventVersion;
import ru.yandex.direct.core.entity.communication.model.CommunicationEventVersionStatus;
import ru.yandex.direct.core.entity.communication.repository.CommunicationEventVersionsRepository;
import ru.yandex.direct.core.entity.communication.repository.CommunicationEventsRepository;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification;
import ru.yandex.direct.juggler.check.model.CheckTag;
import ru.yandex.direct.juggler.check.model.NotificationRecipient;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectJob;
import ru.yandex.direct.solomon.SolomonPushClient;
import ru.yandex.direct.solomon.SolomonPushClientException;
import ru.yandex.direct.solomon.SolomonUtils;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.utils.InterruptedRuntimeException;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YqlQuery;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.misc.io.ClassPathResourceInputStreamSource;
import ru.yandex.monlib.metrics.labels.Labels;

import static java.util.stream.Collectors.collectingAndThen;
import static ru.yandex.direct.communication.CommunicationHelper.parseUsers;
import static ru.yandex.direct.core.entity.communication.model.CommunicationEventVersionStatus.ABORTED;
import static ru.yandex.direct.core.entity.communication.model.CommunicationEventVersionStatus.ABORTING;
import static ru.yandex.direct.core.entity.communication.model.CommunicationEventVersionStatus.ACTIVATING;
import static ru.yandex.direct.core.entity.communication.model.CommunicationEventVersionStatus.ACTIVE;
import static ru.yandex.direct.core.entity.communication.model.CommunicationEventVersionStatus.ARCHIVED;
import static ru.yandex.direct.core.entity.communication.model.CommunicationEventVersionStatus.NEED_ABORT;
import static ru.yandex.direct.core.entity.communication.model.CommunicationEventVersionStatus.NEED_APPROVE;
import static ru.yandex.direct.core.entity.communication.model.CommunicationEventVersionStatus.NEW_;
import static ru.yandex.direct.core.entity.communication.model.CommunicationEventVersionStatus.PREPARING;
import static ru.yandex.direct.core.entity.communication.model.CommunicationEventVersionStatus.READY;
import static ru.yandex.direct.ytwrapper.YtUtils.CONTENT_REVISION_ATTR;
import static ru.yandex.direct.ytwrapper.YtUtils.EXPIRATION_TIME_ATTR;

@JugglerCheck(ttl = @JugglerCheck.Duration(hours = 3),
        tags = {CheckTag.DIRECT_PRIORITY_2, CheckTag.YT},
        notifications = {
                @OnChangeNotification(recipient = {NotificationRecipient.LOGIN_A_DUBOV},
                        status = {JugglerStatus.OK, JugglerStatus.WARN, JugglerStatus.CRIT},
                        method = NotificationMethod.TELEGRAM)
        },
        needCheck = ProductionOnly.class
)
@Hourglass(periodInSeconds = /*every 5 minutes*/ 300)
@ParametersAreNonnullByDefault
public class CommunicationEventPreparingJob extends DirectJob {

    private static final Logger logger = LoggerFactory.getLogger(CommunicationEventPreparingJob.class);
    private static final String QUERY_PATH = "communication/events_by_user_list.sql";

    private final YtProvider ytProvider;
    private final CommunicationEventsRepository communicationEventsRepository;
    private final CommunicationEventVersionsRepository communicationEventVersionsRepository;
    private final SolomonPushClient solomonPushClient;
    private final YtCluster defaultYtCluster;
    private final Map<YtCluster, CommunicationEventSenderParam> paramByCluster;

    @Autowired
    public CommunicationEventPreparingJob(
            YtProvider ytProvider,
            CommunicationEventsRepository communicationEventsRepository,
            CommunicationEventVersionsRepository communicationEventVersionsRepository,
            SolomonPushClient solomonPushClient,
            CommunicationEventSenderParametersSource parametersSource
    ) {
        this.ytProvider = ytProvider;
        this.communicationEventsRepository = communicationEventsRepository;
        this.communicationEventVersionsRepository = communicationEventVersionsRepository;
        this.solomonPushClient = solomonPushClient;
        paramByCluster = StreamEx.of(parametersSource.getAllParamValues())
                .filter(p -> p.getPreparingFolder() != null)
                .toMap(CommunicationEventSenderParam::getYtCluster, Function.identity());
        defaultYtCluster = EntryStream.of(paramByCluster)
                .filterValues(CommunicationEventSenderParam::isDefaultQueue)
                .keys()
                .findFirst()
                .orElseThrow();
    }

    @Override
    public void execute() {
        checkAndUpdateStatuses();
        handlePreparing();
    }

    private void handlePreparing() {
        var readyVersions = communicationEventVersionsRepository
                .getVersionByStatusesAndEventType(List.of(READY), CommunicationEventType.SINGLE_LAUNCH);
        for (var version : readyVersions) {
            if (!tryPrepareIteration(version)) {
                logger.warn(String.format("Can't prepare event %d iter %d",
                        version.getEventId(), version.getIter()));
            }
            checkAndUpdateStatuses();
        }
    }

    private boolean tryPrepareIteration(CommunicationEventVersion cev) {
        cev.setStatus(PREPARING);
        if (!communicationEventVersionsRepository.update(cev, READY, false)) {
            return false;
        }

        var users = cev.getUsers();
        var userTableHash = cev.getUserTableHash();
        boolean usersByTable = userTableHash != null;

        YtCluster cluster;
        String userExpression;
        String userJoinType;
        if (usersByTable) {
            String[] typeColumnClusterPath = parseUsers(users);
            userJoinType = typeColumnClusterPath[0];
            String userTableColumn = typeColumnClusterPath[1];
            cluster = YtCluster.parse(typeColumnClusterPath[2]);
            users = typeColumnClusterPath[3];
            userExpression = String.format("(select %s from $users)", userTableColumn);
        } else {
            userJoinType = "login";
            cluster = defaultYtCluster;
            userExpression = "String::SplitToList($users, ',')";
        }

        var cypress = ytProvider.get(cluster).cypress();
        if (usersByTable) {
            // Проверяем валидность таблички с пользователями
            var yPath = YPath.simple(users);
            String errorMessage;
            if (!cypress.exists(yPath)) {
                errorMessage = String.format("Table %s:%s is not found.", cluster.getName(), users);
                logger.error(errorMessage);
                cev.setStatus(NEW_);
                cev.setUsers(null);
                cev.setUserTableHash(null);
                communicationEventVersionsRepository.update(cev, PREPARING);
                return false;
            }
            var contentRevision = cypress.get(yPath.attribute(CONTENT_REVISION_ATTR)).longValue() + "";
            if (!contentRevision.equals(userTableHash)) {
                errorMessage = String.format("Table %s:%s was changed.", cluster.getName(), users);
                logger.error(errorMessage);
                cev.setUserTableHash(contentRevision);
                cev.setStatus(NEED_APPROVE);
                communicationEventVersionsRepository.update(cev, PREPARING);
                return false;
            }
        }

        // Выбираем куда положить данные на отправку
        var tempFolder = YPath.simple(paramByCluster.get(cluster).getPreparingFolder());
        var tempEventsPath = tempFolder.child(generateTableName(cev, false));
        var rollbackPath = tempFolder.child(generateTableName(cev, true));
        if (cypress.exists(tempEventsPath)) {
            logger.warn(String.format("Table for send (%s:%s) is already exist. Remove it.",
                    cluster.getName(), tempEventsPath.toString()));
            cypress.remove(tempEventsPath);
        }
        if (cypress.exists(rollbackPath)) {
            logger.warn(String.format("Rollback table (%s:%s) is already exist. Remove it.",
                    cluster.getName(), rollbackPath.toString()));
            cypress.remove(rollbackPath);
        }

        // Сгенерировать данные на отправку и откат
        List sqlParams = new ArrayList();
        sqlParams.add(tempEventsPath.toString());
        sqlParams.add(rollbackPath.toString());
        sqlParams.add(cev.getEventId());
        sqlParams.add(cev.getStartTime().toEpochSecond(OffsetDateTime.now().getOffset()));
        sqlParams.add(cev.getExpired().toEpochSecond(OffsetDateTime.now().getOffset()));
        sqlParams.add(cev.getTitle());
        sqlParams.add(cev.getText());
        sqlParams.add(cev.getButtonText());
        sqlParams.add(cev.getButtonHref());
        sqlParams.add(cev.getImageHref());
        sqlParams.add(users);
        String query = String.format(
                String.join("\n", new ClassPathResourceInputStreamSource(QUERY_PATH).readLines()),
                userExpression,
                userJoinType);
        try (var ignore = Trace.current().profile("communication:prepare")) {
            ytProvider.getOperator(cluster).yqlExecute(new YqlQuery(query, sqlParams.toArray())
                    .withTitle("communication:prepare"));
        }

        if (!cypress.exists(tempEventsPath)) {
            logger.error(String.format("Table for send (%s:%s) wasn't created",
                    cluster.getName(), tempEventsPath.toString()));
            cev.setStatus(READY);
            communicationEventVersionsRepository.update(cev, PREPARING, false);
            return false;
        } else {
            cypress.set(tempEventsPath.attribute(EXPIRATION_TIME_ATTR),
                    cev.getExpired().plusHours(6).toInstant(ZoneOffset.UTC).toEpochMilli());
        }
        if (!cypress.exists(rollbackPath)) {
            logger.error(String.format("Table for send (%s:%s) wasn't created",
                    cluster.getName(), rollbackPath.toString()));
            cev.setStatus(READY);
            communicationEventVersionsRepository.update(cev, PREPARING, false);
            return false;
        } else {
            cypress.set(rollbackPath.attribute(EXPIRATION_TIME_ATTR),
                    cev.getExpired().plusHours(6).toInstant(ZoneOffset.UTC).toEpochMilli());
        }

        // Копировать на отправку
        YPath pathTo = YPath.simple(paramByCluster.get(cluster).getQueueFolder())
                .child(generateTableName(cev, false));
        cypress.move(tempEventsPath, pathTo);
        if (!cypress.exists(pathTo)) {
            logger.error(String.format("Table for send (%s:%s) wasn't moved to sending queue",
                    cluster.getName(), pathTo.toString()));
            cev.setStatus(READY);
            communicationEventVersionsRepository.update(cev, PREPARING, false);
            return false;
        } else {
            cypress.set(pathTo.attribute(EXPIRATION_TIME_ATTR),
                    cev.getExpired().plusHours(6).toInstant(ZoneOffset.UTC).toEpochMilli());
        }

        cev.setCluster(cluster.getName());
        cev.setEventsTablePath(pathTo.toString());
        cev.setUserTableHash(null);
        cev.setRollbackEventsTablePath(rollbackPath.toString());
        cev.setStatus(ACTIVATING);
        if (!communicationEventVersionsRepository.update(cev, PREPARING)) {
            return false;
        }
        var startTime = cev.getStartTime();
        if (startTime.isBefore(LocalDateTime.now())) {
            startTime = LocalDateTime.now();
        }
        if (communicationEventsRepository
                .getCommunicationEventsByIds(List.of(cev.getEventId()))
                .stream().findAny()
                .map(CommunicationEvent::getActivateTime)
                .orElse(startTime.plusDays(1))
                .isBefore(startTime)) {
            startTime = null;
        }
        communicationEventsRepository.activateCommunicationEvent(cev.getEventId(), startTime);
        return true;
    }

    private void checkAndUpdateStatuses() {
        var statuses = List.of(ACTIVATING, ABORTING, NEED_ABORT);
        var processes = communicationEventVersionsRepository
                .getVersionsByStatuses(statuses);
        for (CommunicationEventVersion cev : processes) {
            var status = cev.getStatus();
            if (NEED_ABORT.equals(status)) {
                if (!tryAbortIteration(cev)) {
                    logger.warn(String.format("Can't abort event %d iter %d",
                            cev.getEventId(), cev.getIter()));
                }
            } else {
                tryCheckFinishedProcess(cev);
            }
        }
        sendMetricsToSolomon();
    }

    private void sendMetricsToSolomon() {
        var allStatuses = List.of(CommunicationEventVersionStatus.values());
        var statusesForAgeCalculating = new ArrayList<>(allStatuses);
        statusesForAgeCalculating.removeAll(List.of(
                // Для этих статусов считать возраст не имеет смысла,
                // потому что он может быть сколь угодно большим и это норма.
                NEW_, ACTIVE, ABORTED, ARCHIVED
        ));

        var statusToCount = communicationEventVersionsRepository.getStatusesCount();
        var statusToIterCount = StreamEx.of(allStatuses)
                .toMap(Function.identity(), status -> statusToCount
                        .getOrDefault(status, Pair.of(0, 0))
                        .getLeft()
                );
        var statusToEventCount = StreamEx.of(allStatuses)
                .toMap(Function.identity(), status -> statusToCount
                        .getOrDefault(status, Pair.of(0, 0))
                        .getRight()
                );

        var statusToLastChange = StreamEx.of(
                communicationEventVersionsRepository.getVersionsByStatuses(statusesForAgeCalculating))
                .mapToEntry(CommunicationEventVersion::getStatus, CommunicationEventVersion::getLastChangeStatus)
                .grouping(collectingAndThen(Collectors.minBy(String::compareTo), Optional::get));
        var now = Instant.now();
        var statusToAge = StreamEx.of(statusesForAgeCalculating)
                .mapToEntry(Function.identity(), status ->
                        statusToLastChange.getOrDefault(status, now.getEpochSecond() + ""))
                .mapValues(Long::parseLong)
                .mapValues(s -> now.getEpochSecond() - s)
                .toMap();

        var registry = SolomonUtils.newPushRegistry("flow", "CommunicationEventPreparingJob");
        EntryStream.of(statusToEventCount)
                .forEach(e -> registry
                        .gaugeInt64("events", Labels.of("status", e.getKey().name()))
                        .set(e.getValue()));
        EntryStream.of(statusToIterCount)
                .forEach(e -> registry
                        .gaugeInt64("iterations", Labels.of("status", e.getKey().name()))
                        .set(e.getValue()));
        EntryStream.of(statusToAge)
                .forEach(e -> registry
                        .gaugeInt64("age", Labels.of("status", e.getKey().name()))
                        .set(e.getValue()));
        try {
            long timestampMillis = System.currentTimeMillis();
            timestampMillis -= timestampMillis % (60 * 1000);
            solomonPushClient.sendMetrics(registry, timestampMillis);
        } catch (SolomonPushClientException e) {
            logger.error("Got exception on sending metrics", e);
        }
    }

    private boolean tryAbortIteration(CommunicationEventVersion cev) {
        try {
            YtCluster cluster = YtCluster.parse(cev.getCluster());
            String pathForSending = paramByCluster.get(cluster).getQueueFolder();
            YPath pathTo = YPath.simple(pathForSending)
                    .child(generateTableName(cev, true));
            YPath pathFrom = YPath.simple(cev.getRollbackEventsTablePath());

            var cypress = ytProvider.get(cluster).cypress();
            if (cypress.exists(pathTo)) {
                logger.warn("Data for aborting is already exist in queue.");
            } else {
                cypress.copy(pathFrom, pathTo);
            }

            cev.setStatus(ABORTING);
            cev.setRollbackEventsTablePath(pathTo.toString());
            if (communicationEventVersionsRepository.update(cev, NEED_ABORT)) {
                cypress.remove(pathFrom);
            } else {
                logger.warn("Can't update iteration info.");
            }
            return true;
        } catch (InterruptedRuntimeException ex) {
            Thread.currentThread().interrupt();
            throw ex;
        } catch (RuntimeException ex) {
            logger.error("Error while aborting", ex);
        }
        return false;
    }

    private static String generateTableName(CommunicationEventVersion cev, boolean isAborting) {
        StringBuilder sb = new StringBuilder("event_")
                .append(cev.getEventId())
                .append("_iter_")
                .append(cev.getIter());
        if (isAborting) {
            sb.append("_rollback");
        }
        return sb.toString();
    }

    private void tryCheckFinishedProcess(CommunicationEventVersion cev) {
        try {
            CommunicationEventVersionStatus nextStatus;
            ModelProperty<CommunicationEventVersion, String> tablePathProperty;
            List<ModelProperty<CommunicationEventVersion, String>> clearingProperties = new ArrayList<>();
            if (ACTIVATING.equals(cev.getStatus())) {
                nextStatus = ACTIVE;
                tablePathProperty = CommunicationEventVersion.EVENTS_TABLE_PATH;
            } else {
                nextStatus = ABORTED;
                tablePathProperty = CommunicationEventVersion.ROLLBACK_EVENTS_TABLE_PATH;
                clearingProperties.add(CommunicationEventVersion.CLUSTER);
            }
            clearingProperties.add(tablePathProperty);

            var path = YPath.simple(tablePathProperty.get(cev));
            if (ytProvider.get(YtCluster.parse(cev.getCluster()))
                    .cypress().exists(path)) {
                return;
            }
            CommunicationEventVersionStatus currentStatus = cev.getStatus();
            cev.setStatus(nextStatus);
            clearingProperties.forEach(prop -> prop.set(cev, null));
            communicationEventVersionsRepository.update(cev, currentStatus);
        } catch (InterruptedRuntimeException ex) {
            Thread.currentThread().interrupt();
            throw ex;
        } catch (RuntimeException ex) {
            logger.error("Error while check sending finishing", ex);
            return;
        }
    }
}
