package ru.yandex.direct.jobs.communication;

import java.time.Duration;
import java.time.Instant;
import java.util.Collection;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod;
import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.communication.CommunicationClient;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification;
import ru.yandex.direct.juggler.check.model.CheckTag;
import ru.yandex.direct.juggler.check.model.NotificationRecipient;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectParameterizedJob;
import ru.yandex.direct.scheduler.support.ParameterizedBy;
import ru.yandex.direct.solomon.SolomonPushClient;
import ru.yandex.direct.solomon.SolomonPushClientException;
import ru.yandex.direct.solomon.SolomonUtils;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YtOperator;
import ru.yandex.direct.ytwrapper.model.YtTable;
import ru.yandex.inside.yt.kosher.cypress.CypressNodeType;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;
import ru.yandex.monlib.metrics.labels.Labels;

import static java.util.Collections.emptyList;
import static ru.yandex.direct.common.db.PpcPropertyNames.lastSendedCommunicationEventsTableModificationTime;
import static ru.yandex.direct.communication.CommunicationHelper.sendEventsFromTable;

@JugglerCheck(ttl = @JugglerCheck.Duration(hours = 3),
        tags = {CheckTag.DIRECT_PRIORITY_2, CheckTag.YT},
        notifications = {
                @OnChangeNotification(recipient = {NotificationRecipient.LOGIN_A_DUBOV},
                        status = {JugglerStatus.OK, JugglerStatus.WARN, JugglerStatus.CRIT},
                        method = NotificationMethod.TELEGRAM)
        },
        needCheck = ProductionOnly.class
)
@Hourglass(periodInSeconds = /*every 5 minutes*/ 300)
@ParametersAreNonnullByDefault
@ParameterizedBy(parametersSource = CommunicationEventSenderParametersSource.class)
public class CommunicationEventSenderJob extends DirectParameterizedJob<CommunicationEventSenderParam> {

    private static final Logger logger = LoggerFactory.getLogger(CommunicationEventSenderJob.class);

    private static final String MODIFICATION_TIME_ATTRIBUTE = "modification_time";
    private static final String NAME_ATTRIBUTE = "key";
    private static final String TYPE_ATTRIBUTE = "type";
    private static final String ROW_COUNT_ATTRIBUTE = "row_count";
    private static final SetF<String> ATTRIBUTES = Cf.set(
            MODIFICATION_TIME_ATTRIBUTE, NAME_ATTRIBUTE, TYPE_ATTRIBUTE, ROW_COUNT_ATTRIBUTE);

    private static final long CHUNK_SIZE = 1000;
    private static final int RETRY_COUNT = 3;
    private static final long SECONDS_IN_MINUTE = Duration.ofMinutes(1).toSeconds();
    private static final long SECONDS_IN_HOUR = Duration.ofHours(1).toSeconds();

    private final YtProvider ytProvider;
    private final PpcPropertiesSupport ppcPropertiesSupport;
    private final CommunicationEventSenderParametersSource communicationEventSenderParametersSource;
    private final CommunicationClient communicationClient;
    private final SolomonPushClient solomonPushClient;

    @Autowired
    public CommunicationEventSenderJob(
            YtProvider ytProvider,
            PpcPropertiesSupport ppcPropertiesSupport,
            CommunicationEventSenderParametersSource communicationEventSenderParametersSource,
            CommunicationClient communicationClient,
            SolomonPushClient solomonPushClient
    ) {
        this.ytProvider = ytProvider;
        this.ppcPropertiesSupport = ppcPropertiesSupport;
        this.communicationEventSenderParametersSource = communicationEventSenderParametersSource;
        this.communicationClient = communicationClient;
        this.solomonPushClient = solomonPushClient;
    }

    @Override
    public void execute() {
        var paramString = getParam();
        var lastHandledTableModificationTimeProperty = ppcPropertiesSupport.get(lastSendedCommunicationEventsTableModificationTime(paramString));
        var communicationEventSenderParam = communicationEventSenderParametersSource.convertStringToParam(paramString);
        var ytOperator = ytProvider.getOperator(communicationEventSenderParam.getYtCluster());
        List<YTreeNode> tables = checkQueue(ytOperator, communicationEventSenderParam.getQueueFolder(), lastHandledTableModificationTimeProperty);
        if (tables.isEmpty()) {
            return;
        }
        logger.info(String.format("Try to send %d tables from %s:%s",
                tables.size(),
                communicationEventSenderParam.getYtCluster().getName(),
                communicationEventSenderParam.getQueueFolder()
        ));
        int successCount = 0;
        try {
            for (var table : tables) {
                String path = communicationEventSenderParam.getQueueFolder() + "/"
                        + table.getAttribute(NAME_ATTRIBUTE).get().stringValue();
                if (trySendTable(ytOperator, path, table, lastHandledTableModificationTimeProperty)) {
                    successCount++;
                }
            }
        } catch (RuntimeException ex) {
            logger.error("Exception ", ex);
            setJugglerStatus(JugglerStatus.CRIT, ex.getMessage());
        }
        logger.info(String.format("%d tables are sended of %d candidates", successCount, tables.size()));
    }

    private List<YTreeNode> checkQueue(
            YtOperator ytOperator,
            String folderPath,
            PpcProperty<String> lastModificationTimeProperty
    ) {
        var cypress = ytOperator.getYt().cypress();
        var path =YPath.simple(folderPath);
        if (!cypress.exists(path)) {
            String msg = String.format("Directory %s isn't found on cluster %s",
                    folderPath, ytOperator.getCluster().getName());
            logger.warn(msg);
            setJugglerStatus(JugglerStatus.WARN, msg);
            return emptyList();
        }

        // Собираем все таблички на отправку
        Collection<YTreeNode> nodes = cypress.get(path, ATTRIBUTES).asMap().values();
        List<YTreeNode> tables = StreamEx.of(nodes)
                .filter(node -> node.getAttribute(TYPE_ATTRIBUTE).get().stringValue().equals(CypressNodeType.TABLE.value()))
                .sortedBy(this::getModificationTime)
                .toList();
        if (tables.isEmpty()) {
            String msg = String.format("Queue is empty: %s:%s",
                    ytOperator.getCluster().getName(), folderPath);
            logger.info(msg);
            setJugglerStatus(JugglerStatus.OK, msg);
            sendMetrics(ytOperator.getCluster().getName(), folderPath,
                    0, 0, 0, 0);
            return emptyList();
        }

        // Достаем Modification time таблички, последней взятой в обработку
        var lastModificationTimeString = lastModificationTimeProperty.get();
        if (lastModificationTimeString == null) {
            String msg = String.format("Sending from %s:%s is turn off. " +
                            "Set value %s to property %s to turn on sending.",
                    ytOperator.getCluster().getName(), folderPath,
                    getModificationTime(tables.get(0)).minusMillis(1000).toString(),
                    lastModificationTimeProperty.getName());
            logger.warn(msg);
            setJugglerStatus(JugglerStatus.WARN, msg);
            return emptyList();
        }
        var lastModificationTime = Instant.parse(lastModificationTimeString);

        // Берем только таблички, которые еще не обрабатывали
        List<YTreeNode> queueTables = StreamEx.of(tables)
                .filter(table -> lastModificationTime.isBefore(getModificationTime(table)))
                .toList();

        // Собираем метрики
        var queueSize = sumRowCount(queueTables);
        var inProgressSize = sumRowCount(tables) - queueSize;
        long queueAgeSeconds = getQueueAgeSeconds(queueTables);
        long inProgressAgeSeconds = getQueueAgeSeconds(tables);
        logger.info(String.format("There are %d events in queue. There are %d event are sending.",
                queueSize, inProgressSize));
        logger.info(String.format("Age of queue is %d seconds. Age of sending events is %d seconds.",
                queueAgeSeconds, inProgressAgeSeconds));

        //Отправляем метрики и определяем статус
        sendMetrics(ytOperator.getCluster().getName(), folderPath,
                queueSize, inProgressSize, queueAgeSeconds, inProgressAgeSeconds);
        if (inProgressAgeSeconds > 3 * SECONDS_IN_HOUR) {
            String msg = String.format("Age of sending events is %f hours: %s:%s",
                    1.0 * inProgressAgeSeconds / SECONDS_IN_HOUR, ytOperator.getCluster().getName(), folderPath);
            logger.warn(msg);
            setJugglerStatus(JugglerStatus.CRIT, msg);
        } else if (queueAgeSeconds > SECONDS_IN_HOUR) {
            String msg = String.format("Age of queue is %f hours: %s:%s",
                    1.0 * queueAgeSeconds / SECONDS_IN_HOUR, ytOperator.getCluster().getName(), folderPath);
            logger.warn(msg);
            setJugglerStatus(JugglerStatus.CRIT, msg);
        } else if (inProgressAgeSeconds > SECONDS_IN_HOUR) {
            String msg = String.format("Age of sending events is %f hours: %s:%s",
                    1.0 * inProgressAgeSeconds / SECONDS_IN_HOUR, ytOperator.getCluster().getName(), folderPath);
            logger.warn(msg);
            setJugglerStatus(JugglerStatus.WARN, msg);
        } else if (queueAgeSeconds > 10 * SECONDS_IN_MINUTE) {
            String msg = String.format("Age of queue is %f minutes: %s:%s",
                    1.0 * queueAgeSeconds / SECONDS_IN_MINUTE, ytOperator.getCluster().getName(), folderPath);
            logger.warn(msg);
            setJugglerStatus(JugglerStatus.WARN, msg);
        } else {
            setJugglerStatus(JugglerStatus.OK, String.format(
                    "Age of queue is %d seconds. Age of sending events is %d seconds.",
                    queueAgeSeconds, inProgressAgeSeconds
            ));
        }
        return queueTables;
    }

    private void sendMetrics(
            String cluster, String path,
            long queueRowCount, long inProgressRowCount,
            long queueAge, long inProgressAge
    ) {
        var registry = SolomonUtils.newPushRegistry(Labels.of(
                "flow", "CommunicationEventSenderJob",
                "yt_cluster", cluster,
                "path", path));

        registry.gaugeInt64("age", Labels.of("state", "inQueue")).set(queueAge);
        registry.gaugeInt64("age", Labels.of("state", "inProgress")).set(inProgressAge);
        registry.gaugeInt64("events", Labels.of("state", "inQueue")).set(queueRowCount);
        registry.gaugeInt64("events", Labels.of("state", "inProgress")).set(inProgressRowCount);
        try {
            long timestampMillis = System.currentTimeMillis();
            timestampMillis -= timestampMillis % (60 * 1000);
            solomonPushClient.sendMetrics(registry, timestampMillis);
        } catch (SolomonPushClientException e) {
            logger.error("Got exception on sending metrics", e);
        }
    }

    private Instant getModificationTime(YTreeNode table) {
        return Instant.parse(table.getAttribute(MODIFICATION_TIME_ATTRIBUTE).get().stringValue());
    }

    private long getQueueAgeSeconds(List<YTreeNode> tables) {
        return tables.stream()
                .findFirst()
                .map(this::getModificationTime)
                .map(i -> Duration.between(i, Instant.now()).toSeconds())
                .orElse(0L);
    }

    private long sumRowCount(List<YTreeNode> tables) {
        return tables.stream()
                .mapToLong(table -> table.getAttribute(ROW_COUNT_ATTRIBUTE).get().longValue())
                .sum();
    }

    private boolean trySendTable(
            YtOperator ytOperator, String path, YTreeNode table,
            PpcProperty<String> lastModificationTimeProperty
    ) {
        var oldPropertyValue = lastModificationTimeProperty.get();
        if (oldPropertyValue == null) {
            String msg = String.format("Sending from %s:%s is turn off",
                    ytOperator.getCluster().getName(), path);
            logger.warn(msg);
            setJugglerStatus(JugglerStatus.WARN, msg);
            return false;
        }
        var lastModificationTime = Instant.parse(oldPropertyValue);
        var modificationTime = getModificationTime(table);
        if (!lastModificationTime.isBefore(modificationTime)) {
            logger.info(String.format("%s:%s is already sending",
                    ytOperator.getCluster().getName(), path));
            return false;
        }
        var newPropertyValue = modificationTime.toString();
        if (!lastModificationTimeProperty.cas(oldPropertyValue, newPropertyValue)) {
            logger.info(String.format("%s:%s has just started sending",
                    ytOperator.getCluster().getName(), path));
            return false;
        }
        YtTable ytTable = new YtTable(path);
        final long rowCount = ytOperator.readTableRowCount(ytTable);
        for (long startRow = 0L; startRow < rowCount; startRow += CHUNK_SIZE) {
            long endRow = Math.min(startRow + CHUNK_SIZE, rowCount);
            sendEventsFromTable(communicationClient, ytOperator, ytTable, startRow, endRow, RETRY_COUNT);
        }
        ytOperator.removeTable(ytTable);
        return true;
    }
}
