package ru.yandex.direct.jobs.configuration;

import java.util.List;
import java.util.function.Supplier;

import com.google.common.base.Preconditions;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import org.springframework.context.annotation.Lazy;
import org.springframework.context.annotation.Primary;

import ru.yandex.direct.binlogbroker.logbroker_utils.lock.ConsumerLockHolder;
import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.core.entity.moderation.model.AdGroupModerationMeta;
import ru.yandex.direct.core.entity.moderation.model.BaseBannerModerationMeta;
import ru.yandex.direct.core.entity.moderation.model.CampaignModerationMeta;
import ru.yandex.direct.jobs.moderation.config.ErrorLogbrokerConsumerPropertiesHolder;
import ru.yandex.direct.jobs.moderation.config.ResponseModerationLogbrokerConsumerPropertiesHolder;
import ru.yandex.direct.jobs.moderation.config.ResponseModerationParameters;
import ru.yandex.direct.jobs.moderation.config.UnparsedLogbrokerConsumerPropertiesHolder;
import ru.yandex.direct.jobs.moderation.debug.DebugReceiveModerationParams;
import ru.yandex.direct.jobs.moderation.processor.ModerationResponseProcessorFilter;
import ru.yandex.direct.logicprocessor.processors.moderation.ModerationStageEnum;
import ru.yandex.direct.tvm.TvmIntegration;
import ru.yandex.direct.tvm.TvmService;
import ru.yandex.direct.ytwrapper.client.YtClusterTypesafeConfigProvider;
import ru.yandex.kikimr.persqueue.auth.Credentials;

import static ru.yandex.direct.jobs.configuration.ModerationConfiguration.MODERATION_CONSUMER_BEAN;
import static ru.yandex.direct.jobs.configuration.ModerationConfiguration.parseReadTopicsGroups;
import static ru.yandex.direct.logicprocessor.configuration.EssLogicProcessorConfiguration.MODERATION_TVM_INTEGRATION_BEAN_NAME;
import static ru.yandex.direct.logicprocessor.configuration.EssLogicProcessorConfiguration.MODERATION_TVM_LOGBROKER_CREDENTIALS_SUPPLIER_BEAN_NAME;

@Configuration
@Import({
        DebugJobRunnerConfiguration.class,
})
public class DebugReceiveModerationJobRunnerConfiguration {

    private static final String DEFAULT_LOGBROKER_CONF_BRANCH = "moderation_service.logbroker";

    private static DebugReceiveModerationParams appParams;

    public static void initializeParameters(DebugReceiveModerationParams appParams) {
        DebugReceiveModerationJobRunnerConfiguration.appParams = appParams;
    }

    // Lazy(false) - Автоматическая проверка, что метод не забыли позвать. Если захотелось отключить эту проверку, то
    // стоит совсем избавиться от этой конфигурации.
    @Bean
    @Lazy(false)
    public DebugReceiveModerationParams moderationParams() {
        Preconditions.checkState(appParams != null,
                "You forgot to call %s.initializeParameters()", getClass().getName());
        return appParams;
    }

    @Bean(name = MODERATION_TVM_LOGBROKER_CREDENTIALS_SUPPLIER_BEAN_NAME)
    @Primary
    public Supplier<Credentials> getModerationLogbrokerCredentialsSupplier(DirectConfig directConfig,
                                                                           DebugReceiveModerationParams moderationParams,
                                                                           @Qualifier(MODERATION_TVM_INTEGRATION_BEAN_NAME) TvmIntegration tvmIntegration) {
        var moderationStage = ModerationStageEnum.fromValue(moderationParams.essTag);
        var tvmServiceName = moderationStage == null ?
                directConfig.getString("moderation_service.logbroker.tvm_service_name")
                : directConfig.getString(String.format("moderation_service.debug.logbroker.%s.tvm_service_name",
                moderationStage.getValue()));
        var tvmService = TvmService.fromStringStrict(tvmServiceName);
        return () -> {
            String serviceTicket = tvmIntegration.getTicket(tvmService);
            return Credentials.tvm(serviceTicket);
        };
    }

    @Bean
    public ConsumerLockHolder consumerLockHolder(DirectConfig directConfig,
                                                 DebugReceiveModerationParams moderationParams,
                                                 @Value("${moderation_service.debug.yt_lock.cluster_name}") String clusterName,
                                                 @Value("${moderation_service.debug.yt_lock.lock_path_prefix}") String lockPathPrefixName) {
        var moderationStage = ModerationStageEnum.fromValue(moderationParams.essTag);
        var ytClusterConfigProvider = new YtClusterTypesafeConfigProvider(directConfig.getConfig().getConfig("yt"));
        var possibleConsumers = moderationStage == null ?
                directConfig.getStringList("moderation_service.debug.logbroker.consumers")
                : directConfig.getStringList(String.format("moderation_service.debug.logbroker.%s.consumers",
                moderationStage.getValue()));
        return new ConsumerLockHolder(ytClusterConfigProvider, clusterName, lockPathPrefixName, possibleConsumers);
    }

    @Primary
    @Bean(MODERATION_CONSUMER_BEAN)
    public String consumerName(ConsumerLockHolder consumerLockHolder) {
        var consumerNameOptional = consumerLockHolder.getConsumerName();
        if (consumerNameOptional.isEmpty()) {
            throw new IllegalStateException("All consumers are busy");
        }
        return consumerNameOptional.get();
    }


    @Primary
    @Bean
    public ResponseModerationLogbrokerConsumerPropertiesHolder responseModerationLogbrokerReaderConfig(
            DirectConfig directConfig,
            @Qualifier(MODERATION_CONSUMER_BEAN) String consumerName,
            DebugReceiveModerationParams moderationParams) {
        var logbrokerConf = directConfig.getBranch(DEFAULT_LOGBROKER_CONF_BRANCH);
        var stageConf = getStageConf(moderationParams, directConfig);

        List<DirectConfig> topicsConfigs = stageConf.getConfigList("response.topics");
        return ResponseModerationLogbrokerConsumerPropertiesHolder.builder()
                .withConsumerName(consumerName)
                .withHost(stageConf.getString("host"))
                .withLogbrokerDataTimeoutSec(logbrokerConf.getInt("data_timeout"))
                .withInitTimeoutSec(logbrokerConf.getInt("init_timeout"))
                .withRetries(logbrokerConf.getInt("retries"))
                .withReadDataAfterTimestampMs(moderationParams.getReadDataSentAfterTimestampMs())
                .withReadTopicsGroups(parseReadTopicsGroups(topicsConfigs))
                .build();
    }

    @Primary
    @Bean
    public ErrorLogbrokerConsumerPropertiesHolder errorResponseModerationLogbrokerReaderConfig(
            DirectConfig directConfig,
            @Qualifier(MODERATION_CONSUMER_BEAN) String consumerName,
            DebugReceiveModerationParams moderationParams) {
        var logbrokerConf = directConfig.getBranch(DEFAULT_LOGBROKER_CONF_BRANCH);
        var stageConf = getStageConf(moderationParams, directConfig);

        return ErrorLogbrokerConsumerPropertiesHolder.builder()
                .withConsumerName(consumerName)
                .withHost(stageConf.getString("host"))
                .withLogbrokerDataTimeoutSec(logbrokerConf.getInt("data_timeout"))
                .withInitTimeoutSec(logbrokerConf.getInt("init_timeout"))
                .withRetries(logbrokerConf.getInt("retries"))
                .withGroups(stageConf.getIntList("error.group"))
                .withReadDataAfterTimestampMs(moderationParams.getReadDataSentAfterTimestampMs())
                .withReadTopic(stageConf.getString("error.topic"))
                .build();
    }

    @Primary
    @Bean
    public UnparsedLogbrokerConsumerPropertiesHolder unparsedResponseModerationLogbrokerReaderConfig(
            DirectConfig directConfig,
            @Qualifier(MODERATION_CONSUMER_BEAN) String consumerName,
            DebugReceiveModerationParams moderationParams) {
        var logbrokerConf = directConfig.getBranch(DEFAULT_LOGBROKER_CONF_BRANCH);
        var stageConf = getStageConf(moderationParams, directConfig);

        return UnparsedLogbrokerConsumerPropertiesHolder.builder()
                .withConsumerName(consumerName)
                .withHost(stageConf.getString("host"))
                .withLogbrokerDataTimeoutSec(logbrokerConf.getInt("data_timeout"))
                .withInitTimeoutSec(logbrokerConf.getInt("init_timeout"))
                .withRetries(logbrokerConf.getInt("retries"))
                .withGroups(stageConf.getIntList("unparsed.group"))
                .withReadDataAfterTimestampMs(moderationParams.getReadDataSentAfterTimestampMs())
                .withReadTopic(stageConf.getString("unparsed.topic"))
                .build();
    }

    private DirectConfig getStageConf(DebugReceiveModerationParams params, DirectConfig conf) {
        var moderationStage = ModerationStageEnum.fromValue(params.essTag);
        return moderationStage == null ?
                conf.getBranch(DEFAULT_LOGBROKER_CONF_BRANCH)
                : conf.getBranch(String.format("moderation_service.debug.logbroker.%s", moderationStage.getValue()));
    }


    @Primary
    @Bean
    public ResponseModerationParameters responseModerationParameters() {
        return new ResponseModerationParameters.Builder()
                .setLogbrokerNoCommit(true)
                .build();
    }

    @Primary
    @Bean
    public ModerationResponseProcessorFilter moderationResponseProcessorFilter(DebugReceiveModerationParams params) {
        ModerationResponseProcessorFilter filter = ModerationResponseProcessorFilter.doNothing();
        if (!params.campaignIds.isEmpty()) {
            filter.addFilter(r -> {
                if (r.getMeta() instanceof CampaignModerationMeta) {
                    CampaignModerationMeta campMeta = (CampaignModerationMeta) r.getMeta();
                    return params.campaignIds.contains(campMeta.getCampaignId());
                }
                return false;
            });
        }
        if (!params.adgroupIds.isEmpty()) {
            filter.addFilter(r -> {
                if (r.getMeta() instanceof AdGroupModerationMeta) {
                    AdGroupModerationMeta adGroupMeta = (AdGroupModerationMeta) r.getMeta();
                    return params.adgroupIds.contains(adGroupMeta.getAdGroupId());
                }
                return false;
            });
        }
        if (!params.bannerIds.isEmpty()) {
            filter.addFilter(r -> {
                if (r.getMeta() instanceof BaseBannerModerationMeta) {
                    BaseBannerModerationMeta bannerMeta = (BaseBannerModerationMeta) r.getMeta();
                    return params.bannerIds.contains(bannerMeta.getBannerId());
                }
                return false;
            });
        }
        if (!params.essTag.isEmpty()) {
            filter.addFilter(r -> r.getAttributes() != null && params.essTag.equals(r.getAttributes().getBetaPort()));
        }

        filter.addFilter(r -> r.getUnixtime() > params.getReadDataSentAfterTimestampMs());
        return filter;
    }
}
