package ru.yandex.direct.jobs.configuration;

import java.util.Set;

import com.yandex.ydb.table.TableClient;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.ApplicationContext;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Condition;
import org.springframework.context.annotation.ConditionContext;
import org.springframework.context.annotation.Conditional;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import org.springframework.context.annotation.Lazy;
import org.springframework.core.type.AnnotatedTypeMetadata;

import ru.yandex.direct.common.spring.OnlyForTestingCondition;
import ru.yandex.direct.configuration.DirectHourglassConfiguration;
import ru.yandex.direct.configuration.HourglassYdbStorageConfiguration;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.env.Environment;
import ru.yandex.direct.hourglass.InstanceId;
import ru.yandex.direct.hourglass.InstanceMeta;
import ru.yandex.direct.hourglass.MonitoringWriter;
import ru.yandex.direct.hourglass.SchedulerService;
import ru.yandex.direct.hourglass.implementations.ThreadsHierarchy;
import ru.yandex.direct.hourglass.storage.Storage;
import ru.yandex.direct.hourglass.updateschedule.ScheduleUpdateService;
import ru.yandex.direct.hourglass.updateschedule.SchedulerInstancesRepository;
import ru.yandex.direct.hourglass.ydb.monitoring.YdbSessionMonitoring;
import ru.yandex.direct.hourglass.ydb.schedulerinstances.YdbScheduleInstancesRepository;
import ru.yandex.direct.hourglass.ydb.storage.YdbStorageImpl;
import ru.yandex.direct.liveresource.LiveResourceWatcher;
import ru.yandex.direct.scheduler.hourglass.ScheduleRecordListProvider;
import ru.yandex.direct.scheduler.hourglass.TaskFactory;
import ru.yandex.direct.scheduler.hourglass.implementations.HourglassScheduler;
import ru.yandex.direct.scheduler.hourglass.implementations.TaskParameterizer;
import ru.yandex.direct.scheduler.hourglass.implementations.monitoring.juggler.CheckInSchdedulerPingListener;
import ru.yandex.direct.scheduler.hourglass.implementations.monitoring.juggler.HourglassJugglerSender;
import ru.yandex.direct.scheduler.hourglass.implementations.monitoring.juggler.UptimeSchedulerPingListener;
import ru.yandex.monlib.metrics.registry.MetricRegistry;

import static ru.yandex.direct.configuration.DirectHourglassConfiguration.INSTANCES_REPOSITORY_BEAN_NAME;
import static ru.yandex.direct.configuration.DirectHourglassConfiguration.STORAGE_BEAN_NAME;
import static ru.yandex.direct.configuration.DirectHourglassConfiguration.VERSION_BEAN_NAME;
import static ru.yandex.direct.configuration.YdbConfiguration.HOURGLASS_YDB_TABLE_CLIENT_BEAN;
import static ru.yandex.direct.env.EnvironmentType.DEV7;
import static ru.yandex.direct.env.EnvironmentType.DEVTEST;
import static ru.yandex.direct.env.EnvironmentType.PRESTABLE;
import static ru.yandex.direct.env.EnvironmentType.PRODUCTION;
import static ru.yandex.direct.env.EnvironmentType.TESTING;
import static ru.yandex.direct.jobs.JobSchedulerParams.SCHEDULER_VERSION_PARAM;

@Configuration
@Import({
        DirectHourglassConfiguration.class,
        HourglassYdbStorageConfiguration.class,
})
public class HourglassConfiguration {

    @Bean(STORAGE_BEAN_NAME)
    public Storage ydbStorage(YdbStorageImpl storage) {
        return storage;
    }

    @Bean(INSTANCES_REPOSITORY_BEAN_NAME)
    public SchedulerInstancesRepository schedulerInstancesRepository(YdbScheduleInstancesRepository repository) {
        return repository;
    }

    @Bean(VERSION_BEAN_NAME)
    public String version(@Value("${" + SCHEDULER_VERSION_PARAM + "}") String version) {
        return version;
    }

    @Bean
    @Conditional(OnlyForTestingCondition.class)
    public UptimeSchedulerPingListener uptimeSchedulerPingListener(HourglassJugglerSender hourglassJugglerSender) {
        return new UptimeSchedulerPingListener(hourglassJugglerSender);
    }

    @Bean
    public CheckInSchdedulerPingListener checkInSchdedulerPingListener(HourglassJugglerSender hourglassJugglerSender,
                                                                       InstanceId instanceId,
                                                                       InstanceMeta instanceMeta) {
        return new CheckInSchdedulerPingListener(hourglassJugglerSender, instanceId, instanceMeta);
    }

    @Bean
    public TaskParameterizer taskParameterizer(ApplicationContext context, ShardHelper shardHelper) {
        return new TaskParameterizer(context, shardHelper.dbShards());
    }

    @Bean
    public HourglassScheduler hourglassScheduler(ScheduleRecordListProvider scheduleRecordListProvider,
                                                 TaskFactory taskFactory,
                                                 SchedulerService schedulerService,
                                                 ThreadsHierarchy threadsHierarchy,
                                                 @Qualifier("schedulerVersionWatcher") LiveResourceWatcher watcher,
                                                 VersionMismatchTerminator mismatchTerminator,
                                                 ScheduleUpdateService scheduleUpdateService) {
        HourglassScheduler hourglassScheduler =
                new HourglassScheduler(scheduleRecordListProvider, taskFactory, schedulerService,
                        scheduleUpdateService);

        threadsHierarchy.addUncaughtExceptionHandler((a, b) -> hourglassScheduler.stop());

        watcher.addListener(event -> mismatchTerminator.terminateIfVersionMismatch(event.getCurrentContent()));
        watcher.watch();

        return hourglassScheduler;
    }

    @Bean
    @Lazy(false)
    @Conditional(YdbSessionMonitoringCondition.class)
    public YdbSessionMonitoring sessionPoolStatsMonitoring(
            MetricRegistry metricRegistry,
            MonitoringWriter monitoringWriter,
            ThreadsHierarchy threadsHierarchy,
            @Qualifier(HOURGLASS_YDB_TABLE_CLIENT_BEAN) TableClient tableClient
    ) {
        return new YdbSessionMonitoring(metricRegistry, monitoringWriter, threadsHierarchy, tableClient);
    }


    private static class YdbSessionMonitoringCondition implements Condition {
        @Override
        public boolean matches(ConditionContext context, AnnotatedTypeMetadata metadata) {
            return Set.of(PRODUCTION, PRESTABLE, TESTING, DEVTEST, DEV7).contains(Environment.getCached());
        }
    }
}
