package ru.yandex.direct.jobs.configuration;


import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.nio.file.Files;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.EnumMap;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.Nullable;

import com.mongodb.MongoClient;
import com.mongodb.MongoClientURI;
import com.typesafe.config.Config;
import com.typesafe.config.ConfigFactory;
import org.apache.http.client.HttpClient;
import org.asynchttpclient.AsyncHttpClient;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.BeanCreationException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.ApplicationContext;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import org.springframework.context.annotation.Lazy;
import org.springframework.context.annotation.Primary;
import org.springframework.context.annotation.Scope;
import org.springframework.core.io.Resource;
import org.springframework.core.io.ResourceLoader;
import org.springframework.core.io.support.ResourcePatternResolver;
import org.tmatesoft.svn.core.auth.ISVNAuthenticationManager;
import org.tmatesoft.svn.core.internal.wc.DefaultSVNOptions;
import org.tmatesoft.svn.core.wc.SVNClientManager;
import org.tmatesoft.svn.core.wc.SVNWCUtil;

import ru.yandex.direct.ansiblejuggler.AnsibleWrapperConfiguration;
import ru.yandex.direct.bsexport.configuration.BsExportConfiguration;
import ru.yandex.direct.common.configuration.CommonConfiguration;
import ru.yandex.direct.common.configuration.UacYdbConfiguration;
import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.communication.config.CommunicationConfiguration;
import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.core.configuration.CoreConfiguration;
import ru.yandex.direct.core.entity.autooverdraftmail.AutoOverdraftMailNotificationType;
import ru.yandex.direct.core.entity.bs.export.model.WorkerSpec;
import ru.yandex.direct.core.entity.campaign.container.AffectedCampaignIdsContainer;
import ru.yandex.direct.core.entity.campaign.container.StubAffectedCampaignIdsContainer;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.RequestCampaignAccessibilityCheckerProvider;
import ru.yandex.direct.core.entity.currency.repository.CurrencyRateRepository;
import ru.yandex.direct.core.entity.eventlog.model.DaysLeftNotificationType;
import ru.yandex.direct.core.entity.motivationmail.MotivationMailNotificationType;
import ru.yandex.direct.core.entity.motivationmail.MotivationMailStats;
import ru.yandex.direct.core.entity.uac.grut.GrutContext;
import ru.yandex.direct.core.entity.uac.grut.ThreadLocalGrutContext;
import ru.yandex.direct.core.grut.api.GrutApiProperties;
import ru.yandex.direct.core.grut.api.GrutReplicationApiProperties;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.env.EnvironmentCondition;
import ru.yandex.direct.env.EnvironmentType;
import ru.yandex.direct.env.EnvironmentTypeProvider;
import ru.yandex.direct.i18n.Language;
import ru.yandex.direct.jobs.autooverdraft.AutoOverdraftMailTemplateResolver;
import ru.yandex.direct.jobs.bsexport.BsExportWorkerParam;
import ru.yandex.direct.jobs.bsexport.BsExportWorkerParametersSource;
import ru.yandex.direct.jobs.campaign.paused.daybudget.PausedByDayBudgetCampaignsWarningsMailTemplateResolver;
import ru.yandex.direct.jobs.communication.CommunicationEventSenderParametersSource;
import ru.yandex.direct.jobs.directdb.metrics.HomeDirectDbOperationsMetricProvider;
import ru.yandex.direct.jobs.directdb.repository.OperationRepository;
import ru.yandex.direct.jobs.directdb.service.YqlClasspathObtainerService;
import ru.yandex.direct.jobs.logs.configuration.YtBsExportLogParametersSource;
import ru.yandex.direct.jobs.logs.configuration.YtBsExportLogParametersSourceProduction;
import ru.yandex.direct.jobs.logs.configuration.YtBsExportLogParametersSourceTesting;
import ru.yandex.direct.jobs.moneyoutreminder.MoneyOutReminderMailTemplateResolver;
import ru.yandex.direct.jobs.moneyoutreminder.MoneyOutReminderNotificationType;
import ru.yandex.direct.jobs.motivationemail.MotivationMailTemplateResolver;
import ru.yandex.direct.jobs.ppcdataexport.model.PpcDataExportInitInfo;
import ru.yandex.direct.jobs.recommendations.WorkerInfo;
import ru.yandex.direct.jobs.recommendations.YtLastAccessTsLoader;
import ru.yandex.direct.jobs.statistics.rollbacknotifications.StatRollbacksMailNotificationType;
import ru.yandex.direct.jobs.statistics.rollbacknotifications.StatRollbacksMailTemplateResolver;
import ru.yandex.direct.jobs.util.svn.SvnSshAgentAuthenticationProvider;
import ru.yandex.direct.jobs.verifications.BoidCheckMonitoringMetricsProvider;
import ru.yandex.direct.jobs.walletswarnings.WalletsWarningsMailTemplateResolver;
import ru.yandex.direct.jobs.yt.MysqlYtSyncBackupConfig;
import ru.yandex.direct.jobs.yt.audit.MysqlYtCheckTaskParametersSource;
import ru.yandex.direct.juggler.AsyncHttpJugglerClient;
import ru.yandex.direct.juggler.JugglerAsyncSender;
import ru.yandex.direct.juggler.JugglerSender;
import ru.yandex.direct.juggler.check.configuration.JugglerCheckConfiguration;
import ru.yandex.direct.liveresource.LiveResource;
import ru.yandex.direct.liveresource.LiveResourceFactory;
import ru.yandex.direct.liveresource.LiveResourceReadException;
import ru.yandex.direct.liveresource.provider.LiveResourceFactoryBean;
import ru.yandex.direct.logicprocessor.configuration.EssLogicProcessorConfiguration;
import ru.yandex.direct.samovar.configuration.SamovarLogbrokerConfiguration;
import ru.yandex.direct.solomon.SolomonPushClient;
import ru.yandex.direct.transfermanagerutils.TransferManagerConfig;
import ru.yandex.direct.tvm.TvmIntegration;
import ru.yandex.direct.tvm.TvmService;
import ru.yandex.direct.utils.Interrupts;
import ru.yandex.direct.utils.io.FileUtils;
import ru.yandex.direct.ytcomponents.spring.YtComponentsConfiguration;
import ru.yandex.direct.ytcore.spring.YtCoreConfiguration;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.grut.client.GrutClient;
import ru.yandex.kikimr.persqueue.LogbrokerClientFactory;
import ru.yandex.kikimr.persqueue.auth.Credentials;
import ru.yandex.kikimr.persqueue.consumer.SyncConsumer;
import ru.yandex.kikimr.persqueue.consumer.sync.SyncConsumerConfig;
import ru.yandex.kikimr.persqueue.proxy.ProxyBalancer;
import ru.yandex.startrek.client.Session;
import ru.yandex.startrek.client.StartrekClientBuilder;
import ru.yandex.startrek.client.utils.HttpClientUtils;

import static java.util.Arrays.asList;
import static java.util.Collections.singletonList;
import static java.util.stream.Collectors.toList;
import static org.springframework.beans.factory.config.ConfigurableBeanFactory.SCOPE_PROTOTYPE;
import static ru.yandex.direct.ansiblejuggler.AnsibleWrapperConfiguration.DEFAULT_ANSIBLE_EXECUTION_TIMEOUT;
import static ru.yandex.direct.ansiblejuggler.AnsibleWrapperConfiguration.LINUX_ANSIBLE_PLAYBOOK_PATH;
import static ru.yandex.direct.ansiblejuggler.AnsibleWrapperConfiguration.NULL_PATH;
import static ru.yandex.direct.core.configuration.CoreConfiguration.ACCESSIBLE_CAMPAIGN_CHEKER_PROVIDER;
import static ru.yandex.direct.grid.core.entity.recommendation.RecommendationTablesUtils.CONF_BASE_DIR;
import static ru.yandex.direct.utils.CommonUtils.nvl;

/**
 * Общая часть конфигурации для {@link ru.yandex.direct.jobs.JobsApp} и {@link ru.yandex.direct.jobs.DebugJobRunner}
 */
@Configuration
@Import({
        CommonConfiguration.class,
        CoreConfiguration.class,
        JugglerCheckConfiguration.class,
        EssLogicProcessorConfiguration.class,
        YtComponentsConfiguration.class,
        YtCoreConfiguration.class,
        UserActionLogConfiguration.class,
        ModerationConfiguration.class,
        BsExportConfiguration.class,
        MaintenanceHelpersYdbConfiguration.class,
        SystemMonitoringConfiguration.class,
        SamovarLogbrokerConfiguration.class,
        UacYdbConfiguration.class,
        CommunicationConfiguration.class,
})
public class JobsEssentialConfiguration {
    public static final String DEFAULT_YT_CLUSTER = "DEFAULT_YT_CLUSTER";
    public static final int COUNT_OF_RECOMMENDATION_WORKERS_FOR_EACH_SHARD = 1;
    public static final String YT_LAST_ACCESS_TS_LOADER = "YtLastAccessTsLoader";
    public static final String RECOMMENDATIONS_BASE_DIR_BEAN = "RecommendationsBaseDir";
    public static final String URL_MONITORING_LB_SYNC_CONSUMER_PROVIDER = "urlMonitoringLbSyncConsumer";
    public static final String STARTREK_ROBOT_ADS_AUDIT_TOKEN = "startrekTokenRobotAdsAudit";
    public static final String STARTREK_SESSION_ROBOT_DIRECT_FEATURE = "startrekSessionRobotDirectFeature";
    public static final String STARTREK_HTTP_CLIENT_BEAN = "startrekHttpClient";
    public static final String PPC_DATA_EXPORT_PATH = "/export/ppcdataexport/";
    public static final String AB_PREPARE_DATA_CLUSTERS = "ab-prepare-data-clusters";
    private static final Logger logger = LoggerFactory.getLogger(JobsEssentialConfiguration.class);
    private static final String URL_MONITORING_LB_CONFIG_SECTION_NAME = "url-monitoring-event-handler";
    private final ResourcePatternResolver resourceResolver;

    @Autowired
    private EnvironmentTypeProvider environmentTypeProvider;

    @Autowired
    private ApplicationContext context;

    @Autowired
    public JobsEssentialConfiguration(ResourcePatternResolver resourceResolver) {
        this.resourceResolver = resourceResolver;
    }


    /**
     * Общая часть конфигурации AnsibleWrapper
     *
     * @return новый экземпляр преднастроенного билдера конфигурации
     */
    static AnsibleWrapperConfiguration.Builder getAnsibleWrapperConfBuilder() {
        return new AnsibleWrapperConfiguration.Builder()
                .withExecutionTimeout(DEFAULT_ANSIBLE_EXECUTION_TIMEOUT)
                .withInventoryContent("localhost\n")
                .withAnsibleLogPath(NULL_PATH);
    }

    @Bean(DEFAULT_YT_CLUSTER)
    @Lazy
    public YtCluster defaultYtCluster(DirectConfig config) {
        DirectConfig ytConfig = config.getBranch("yt");
        String defaultCluster = ytConfig.getString("default_cluster");
        return YtCluster.valueOf(defaultCluster.toUpperCase());
    }

    @Bean
    JugglerSender jugglerClient(@Value("${juggler.eventsGateway}") String eventsGateway,
                                AsyncHttpClient asyncHttpClient) {
        return new JugglerAsyncSender(new AsyncHttpJugglerClient(eventsGateway, true, asyncHttpClient));
    }

    /**
     * Конфигурация AnsibleWrapper для linux-серверов c установленным ansible-playbook из yandex-du-ansible-juggler
     * Содержит в себе свежую версию ansible-juggler и чтение OAuth-токена
     */
    @Bean
    public AnsibleWrapperConfiguration ansibleWrapperConfiguration() {
        return getAnsibleWrapperConfBuilder()
                .withAnsiblePlaybookCmd(LINUX_ANSIBLE_PLAYBOOK_PATH)
                .build();
    }

    @Bean
    MotivationMailTemplateResolver motivationMailTemplateResolver(DirectConfig directConfig) {
        DirectConfig motivationMails = directConfig.getBranch("motivation_mails");
        Map<MotivationMailStats.ClientType, Map<Language, Map<MotivationMailNotificationType, String>>>
                templatesMap = new EnumMap<>(MotivationMailStats.ClientType.class);
        for (MotivationMailStats.ClientType clientType : MotivationMailStats.ClientType.values()) {
            DirectConfig clientTypeBranch = motivationMails.getBranch(clientType.toString());
            Map<Language, Map<MotivationMailNotificationType, String>> clientTypeMap = new EnumMap<>(Language.class);
            for (Language lang : Language.values()) {
                DirectConfig langsBranch = clientTypeBranch.getBranch(lang.getLangString());
                Map<MotivationMailNotificationType, String> langsMappings =
                        new EnumMap<>(MotivationMailNotificationType.class);
                for (MotivationMailNotificationType notification : MotivationMailNotificationType.values()) {
                    Optional<String> templateName = langsBranch.findString(notification.name());
                    templateName.filter(str -> !str.isEmpty()).ifPresent(s -> langsMappings.put(notification, s));
                }
                clientTypeMap.put(lang, langsMappings);
            }
            templatesMap.put(clientType, clientTypeMap);
        }
        return new MotivationMailTemplateResolver(templatesMap);
    }

    @Bean
    AutoOverdraftMailTemplateResolver autoOverdraftMailTemplateResolver(DirectConfig directConfig) {
        DirectConfig autoOverdraftMails = directConfig.getBranch("auto_overdraft_mails");
        Map<AutoOverdraftMailNotificationType, String> templatesMap = new HashMap<>();
        for (AutoOverdraftMailNotificationType notification : AutoOverdraftMailNotificationType.values()) {
            Optional<String> templateName = autoOverdraftMails.findString(notification.name());
            templatesMap.put(notification, templateName.orElseThrow(() ->
                    new NoSuchElementException("No templateName found for " + notification.name())));
        }
        return new AutoOverdraftMailTemplateResolver(templatesMap);
    }

    @Bean
    WalletsWarningsMailTemplateResolver walletsWarningsMailTemplateResolver(DirectConfig directConfig) {
        DirectConfig walletsWarningsMails = directConfig.getBranch("wallets_warnings_mails");
        Map<Language, Map<DaysLeftNotificationType, String>> templatesMap = new EnumMap<>(Language.class);
        for (Language lang : Language.values()) {
            DirectConfig langsBranch = walletsWarningsMails.getBranch(lang.getLangString());
            Map<DaysLeftNotificationType, String> langsMappings =
                    new EnumMap<>(DaysLeftNotificationType.class);
            for (DaysLeftNotificationType notification : DaysLeftNotificationType.values()) {
                Optional<String> templateName = langsBranch.findString(notification.name());
                templateName.filter(str -> !str.isEmpty()).ifPresent(s -> langsMappings.put(notification, s));
            }
            templatesMap.put(lang, langsMappings);
        }
        return new WalletsWarningsMailTemplateResolver(templatesMap);
    }

    @Bean
    MoneyOutReminderMailTemplateResolver moneyOutReminderMailTemplateResolver(DirectConfig directConfig) {
        DirectConfig moneyOutReminderMails = directConfig.getBranch("money_out_reminder_mails");
        Map<Language, Map<MoneyOutReminderNotificationType, String>> templatesMap = new EnumMap<>(Language.class);
        for (Language lang : Language.values()) {
            DirectConfig langsBranch = moneyOutReminderMails.getBranch(lang.getLangString());
            Map<MoneyOutReminderNotificationType, String> langsMappings =
                    new EnumMap<>(MoneyOutReminderNotificationType.class);
            for (MoneyOutReminderNotificationType notification : MoneyOutReminderNotificationType.values()) {
                Optional<String> templateName = langsBranch.findString(notification.name());
                templateName.filter(str -> !str.isEmpty()).ifPresent(s -> langsMappings.put(notification, s));
            }
            templatesMap.put(lang, langsMappings);
        }
        return new MoneyOutReminderMailTemplateResolver(templatesMap);
    }

    @Bean
    PausedByDayBudgetCampaignsWarningsMailTemplateResolver pausedByDayBudgetCampaignsWarningsMailTemplateResolver(
            DirectConfig directConfig) {
        DirectConfig pausedByDayBudgetMails = directConfig.getBranch("paused_day_budget_warnings_mails");
        Map<Language, String> templatesMap = new EnumMap<>(Language.class);
        for (var language : Language.values()) {
            pausedByDayBudgetMails.findString(language.getLangString())
                    .filter(str -> !str.isBlank())
                    .ifPresent(str -> templatesMap.put(language, str));
        }
        return new PausedByDayBudgetCampaignsWarningsMailTemplateResolver(templatesMap);
    }

    @Bean
    StatRollbacksMailTemplateResolver statRollbacksMailTemplateResolver(DirectConfig directConfig) {
        DirectConfig statRollbacksMails = directConfig.getBranch("stat_rollbacks_mails");
        Map<Language, Map<StatRollbacksMailNotificationType, String>> templatesMap = new EnumMap<>(Language.class);
        for (Language lang : Language.values()) {
            DirectConfig langsBranch = statRollbacksMails.getBranch(lang.getLangString());
            Map<StatRollbacksMailNotificationType, String> langsMappings =
                    new EnumMap<>(StatRollbacksMailNotificationType.class);
            for (StatRollbacksMailNotificationType notification : StatRollbacksMailNotificationType.values()) {
                Optional<String> templateName = langsBranch.findString(notification.name());
                templateName.filter(str -> !str.isEmpty()).ifPresent(s -> langsMappings.put(notification, s));
            }
            templatesMap.put(lang, langsMappings);
        }
        return new StatRollbacksMailTemplateResolver(templatesMap);
    }

    @Bean
    public RecommendationsYtClustersParametersSource recommendationsYtClustersParametersSource(
            EnvironmentTypeProvider environmentTypeProvider) {
        List<YtCluster> clusters;
        if (environmentTypeProvider.get().isProductionOrPrestable()) {
            clusters = asList(YtCluster.SENECA_SAS, YtCluster.SENECA_VLA, YtCluster.SENECA_MAN);
        } else {
            clusters = singletonList(YtCluster.ZENO);
        }
        return new RecommendationsYtClustersParametersSource(clusters);
    }

    @Bean
    public RecommendationsApplicationParametersSource recommendationsApplicationParametersSource(
            ShardHelper shardHelper) {
        List<WorkerInfo> workerInfos = new ArrayList<>();
        for (int shard : shardHelper.dbShards()) {
            for (int workerNumber = 1; workerNumber <= COUNT_OF_RECOMMENDATION_WORKERS_FOR_EACH_SHARD; workerNumber++) {
                workerInfos.add(new WorkerInfo(shard, workerNumber));
            }
        }
        return new RecommendationsApplicationParametersSource(workerInfos);
    }

    @Bean(name = YT_LAST_ACCESS_TS_LOADER)
    public YtLastAccessTsLoader ytLastAccessTsLoader() {
        return new YtLastAccessTsLoader();
    }

    @Bean(name = RECOMMENDATIONS_BASE_DIR_BEAN)
    public String recommendationsBaseDir(DirectConfig config) {
        return config.getString(CONF_BASE_DIR);
    }

    @Nullable
    @Bean(name = STARTREK_ROBOT_ADS_AUDIT_TOKEN)
    public String startrekTokenRobotAdsAudit(DirectConfig config) {
        try {
            return LiveResourceFactory.get(config.getString("startrek.robot_ads_audit.token_path"))
                    .getContent()
                    .trim();
        } catch (LiveResourceReadException e) {
            // В непродакшен-средах файла может не быть, лучше не замусоривать логи стектрейсами
            logger.warn("Can't get token for robot-ads-audit@");
            return null;
        }
    }

    @Bean(name = STARTREK_HTTP_CLIENT_BEAN)
    public HttpClient startrekHttpClient() {
        return HttpClientUtils.https(
                TimeUnit.SECONDS.toMillis(5),
                TimeUnit.SECONDS.toMillis(5),
                10,
                "Startrek Java Client"
        );
    }

    @Bean
    TransferManagerConfig transferManagerConfig(DirectConfig directConfig) {
        DirectConfig transferManagerBranch = directConfig.getBranch("transfer_manager");
        try {
            return new TransferManagerConfig(
                    new URL(transferManagerBranch.getString("transfer_manager_api")),
                    FileUtils.expandHome(transferManagerBranch.getString("token_path")),
                    transferManagerBranch.getInt("max_retries"));
        } catch (MalformedURLException e) {
            throw new IllegalArgumentException("Unable to parse TM URL", e);
        }
    }

    @Bean
    public AffectedCampaignIdsContainer affectedCampaignIdsContainer() {
        return new StubAffectedCampaignIdsContainer();
    }

    @Bean(name = ACCESSIBLE_CAMPAIGN_CHEKER_PROVIDER)
    public RequestCampaignAccessibilityCheckerProvider accessibilityCheckerProvider() {
        return new RequestCampaignAccessibilityCheckerProvider();
    }

    @Bean
    public DirectExportYtClustersParametersSource directExportYtClustersParametersSource() {
        return new DirectExportYtClustersParametersSource(asList(YtCluster.ARNOLD, YtCluster.HAHN));
    }

    @Bean
    public MysqlYtCheckTaskParametersSource mysqlYtCheckTaskParametersSource(
            EnvironmentTypeProvider environmentTypeProvider, ShardHelper shardHelper) {
        Set<String> dbNames = Stream.concat(
                Stream.of("ppcdict"),
                shardHelper.dbShards().stream().map(shard -> "ppc:" + shard)
        ).collect(Collectors.toSet());

        Set<YtCluster> clusters = environmentTypeProvider.get().isProductionOrPrestable()
                ? Set.of(YtCluster.HAHN, YtCluster.ARNOLD)
                : Set.of(YtCluster.ZENO);

        return new MysqlYtCheckTaskParametersSource(dbNames, clusters);
    }

    @Bean
    public PpcDataExportParametersSource ppcDataExportParametersSource(DirectExportYtClustersParametersSource ytClustersParametersSource) throws IOException {
        List<PpcDataExportInitInfo> ppcDataExportInitInfo
                = getPpcDataExportJobInitInfo(ytClustersParametersSource.getAllParamValues());
        return PpcDataExportParametersSource.from(ppcDataExportInitInfo);
    }

    @Bean
    public YtBsExportLogParametersSource ytBsExportLogParametersSource(EnvironmentTypeProvider envProvider) {
        if (envProvider.get().isProduction()) {
            return new YtBsExportLogParametersSourceProduction();
        } else {
            return new YtBsExportLogParametersSourceTesting();
        }
    }

    @Bean
    public MysqlYtSyncBackupConfig mysqlYtSyncBackupConfig(DirectConfig config,
                                                           EnvironmentTypeProvider environmentTypeProvider) {
        DirectConfig section = config.getBranch("mysql_yt_sync_backup");
        return new MysqlYtSyncBackupConfig(
                section.getString("primary_medium"),
                Duration.ofHours(section.getInt("src_cluster_copy_ttl_hours")),
                Duration.ofDays(section.getInt("dst_cluster_copy_ttl_days"))
        );
    }

    @Bean
    public MongoClient mongoClient(DirectConfig directConfig) {
        try {
            var mongoString = Files.readString(FileUtils.expandHome(
                    directConfig.getBranch("mongo").getString("token_path"))).strip();
            return new MongoClient(new MongoClientURI(mongoString));
        } catch (Exception ex) {
            // в sandbox нет монги, поэтому возвращаем пустой клиент
            // он тут же начнёт коннектиться к localhost:27017 и ругаться, но бин будет создан
            return new MongoClient();
        }
    }

    /**
     * Создаваемый консьюмер нужно явно закрывать для освобожнения ресурсов.
     * Он реализует интерфейс AutoCloseable, поэтому можно это делать через конструкцию try().
     */
    @Bean(name = URL_MONITORING_LB_SYNC_CONSUMER_PROVIDER)
    @Scope(SCOPE_PROTOTYPE)
    public SyncConsumer syncConsumer(DirectConfig directRootConfig, TvmIntegration tvmIntegration) {
        DirectConfig urlMonitoringConfig = directRootConfig.getBranch(URL_MONITORING_LB_CONFIG_SECTION_NAME);
        return Interrupts.failingGet(() -> createSyncConsumer(urlMonitoringConfig, tvmIntegration));
    }

    /**
     * Бин, который периодически отправляет в Solomon метрики, полученные джобой
     * {@link ru.yandex.direct.jobs.verifications.BoidCheckMonitoringJob}
     */
    @Bean
    @Lazy(false)
    public BoidCheckMonitoringMetricsProvider boidCheckMonitoringMetricsProvider(
            DirectExportYtClustersParametersSource parametersSource,
            PpcPropertiesSupport ppcPropertiesSupport) {
        return new BoidCheckMonitoringMetricsProvider(parametersSource, ppcPropertiesSupport);
    }

    @Bean
    public YqlClasspathObtainerService yqlClasspathObtainerService(ResourcePatternResolver resourcePatternResolver,
                                                                   ResourceLoader resourceLoader,
                                                                   PpcPropertiesSupport ppcPropertiesSupport,
                                                                   CurrencyRateRepository currencyRateRepository) {
        return new YqlClasspathObtainerService(
                resourcePatternResolver, resourceLoader, ppcPropertiesSupport, currencyRateRepository
        );
    }

    @Bean
    public OperationRepository operationRepository(YtProvider ytProvider,
                                                   YqlClasspathObtainerService yqlClassPathObtainerService) {
        return new OperationRepository(ytProvider, yqlClassPathObtainerService);
    }

    @Bean
    @Lazy(false)
    public HomeDirectDbOperationsMetricProvider homeDirectDbOperationsMetricProvider(
            OperationRepository operationRepository,
            SolomonPushClient solomonPushClient
    ) {
        return new HomeDirectDbOperationsMetricProvider(
                operationRepository,
                solomonPushClient
        );
    }

    @Bean(AB_PREPARE_DATA_CLUSTERS)
    public List<YtCluster> abPrepareDataClusters(DirectConfig directConfig) {
        return directConfig.getStringList("abt.prepare.clusters").stream()
                .map(YtCluster::valueOf)
                .collect(toList());
    }

    @Bean
    public CommunicationEventSenderParametersSource communicationSenderParametersSource(DirectConfig directConfig) {
        return new CommunicationEventSenderParametersSource(directConfig.getBranch("communication.send"));
    }

    @Bean
    public BsExportWorkerParametersSource bsExportWorkerParametersSource(DirectConfig directConfig,
                                                                         ShardHelper shardHelper) {
        var allowedWorkersSet = Set.copyOf(Optional.ofNullable(directConfig.getStringList("bs_export_worker" +
                ".allowed_workers")).orElse(List.of()));
        var allPossibleWorkersSpec = WorkerSpec.values();
        var suitableWorkersSpec = Arrays.stream(allPossibleWorkersSpec)
                .filter(workerSpecName -> allowedWorkersSet.contains(workerSpecName.name()))
                .collect(toList());
        var shards = shardHelper.dbShards();
        var allParamValues = shards.stream()
                .flatMap(shard -> suitableWorkersSpec.stream()
                        .map(worker -> new BsExportWorkerParam(worker, shard)))
                .collect(Collectors.toList());
        return new BsExportWorkerParametersSource(allParamValues);
    }

    @Bean
    @Lazy
    public SVNClientManager svnClientManager(
            @Value("${svn.user}") String user,
            @Value("${svn.key_file}") String keyFile
    ) {
        ISVNAuthenticationManager authManager = SVNWCUtil.createDefaultAuthenticationManager(
                SVNWCUtil.getDefaultConfigurationDirectory(),
                user,
                (char[]) null,
                FileUtils.expandHome(keyFile).toFile(),
                null,
                false
        );
        authManager.setAuthenticationProvider(new SvnSshAgentAuthenticationProvider(user));

        return SVNClientManager.newInstance(
                new DefaultSVNOptions(),
                authManager
        );
    }

    @Bean
    @Primary
    public GrutApiProperties grutApiProperties(PpcPropertiesSupport ppcPropertiesSupport) {
        return new GrutReplicationApiProperties(ppcPropertiesSupport);
    }

    @Bean
    @Lazy
    public GrutContext grutContext(GrutClient grutClient) {
        return new ThreadLocalGrutContext(grutClient);
    }

    @Bean
    @Primary
    public Session startrekSession(DirectConfig directConfig, EnvironmentType environmentType,
                                   @Nullable @Qualifier(STARTREK_ROBOT_ADS_AUDIT_TOKEN) String startrekToken) {
        // используем тестовый стартрек, если окружение не продовое
        String apiUrl;
        if (environmentType.equals(EnvironmentType.PRODUCTION)) {
            apiUrl = directConfig.getString("startrek.startrek_api_url");
        } else {
            apiUrl = "https://st-api.test.yandex-team.ru";
            startrekToken = nvl(startrekToken, "");
        }
        return StartrekClientBuilder.newBuilder()
                .uri(apiUrl)
                .maxConnections(10)
                .build(startrekToken);
    }

    @Bean(name = STARTREK_SESSION_ROBOT_DIRECT_FEATURE)
    @Lazy
    public Session startrekSessionRobotDirectFeature(
            @Value("${startrek.robot_direct_feature.token_path}") String tokenPath,
            @Value("${startrek.startrek_api_url}") String apiUrl,
            LiveResourceFactoryBean liveResourceFactoryBean) {
        LiveResource token = liveResourceFactoryBean.get(tokenPath);
        return StartrekClientBuilder.newBuilder().uri(apiUrl).build(token.getContent().trim());
    }

    private SyncConsumer createSyncConsumer(DirectConfig directConfig, TvmIntegration tvmIntegration)
            throws InterruptedException {
        TvmService tvmService = TvmService.fromStringStrict(directConfig.getString("tvm_service_name"));
        SyncConsumerConfig config = getConsumerConfig(() -> Credentials.tvm(tvmIntegration.getTicket(tvmService)),
                directConfig);
        String host = directConfig.getString("host");
        ProxyBalancer proxyBalancer = new ProxyBalancer(host);
        long timeoutSec = directConfig.getLong("timeout_sec");

        try {
            SyncConsumer syncConsumer = new LogbrokerClientFactory(proxyBalancer)
                    .syncConsumer(config, timeoutSec, TimeUnit.SECONDS);
            syncConsumer.init();
            return syncConsumer;
        } catch (TimeoutException e) {
            throw new BeanCreationException("Failed to create LB sync consumer using config: " + directConfig, e);
        }
    }

    private SyncConsumerConfig getConsumerConfig(Supplier<Credentials> logBrokerCredentialsSupplier,
                                                 DirectConfig directConfig) {
        String readTopic = directConfig.getString("read_topic");
        String consumerName = directConfig.getString("consumer_name");
        int timeoutSec = directConfig.getInt("timeout_sec");
        return SyncConsumerConfig.builder(singletonList(readTopic), consumerName)
                .setReadDataTimeout(timeoutSec, TimeUnit.SECONDS)
                .setInitTimeout(timeoutSec, TimeUnit.SECONDS)
                .setCredentialsProvider(logBrokerCredentialsSupplier)
                .build();
    }

    /**
     * Получить данные на основе которых будут созданы джобы для выгрузок
     *
     * @param defaultYtClusters - yt кластеры на которых будут работать выгрузки по умолчанию
     * @return список из пути конфига и множества ytClusters для этого конфига
     */
    private List<PpcDataExportInitInfo> getPpcDataExportJobInitInfo(List<YtCluster> defaultYtClusters) throws IOException {
        Resource[] resources = resourceResolver.getResources("classpath:" + PPC_DATA_EXPORT_PATH + "**/*.conf");

        List<PpcDataExportInitInfo> result = new ArrayList<>();
        for (Resource resource : resources) {
            String fullPath = resource.getURL().getPath();
            int classpathStartIndex = fullPath.lastIndexOf(PPC_DATA_EXPORT_PATH);
            String relativePath = "classpath:" + fullPath.substring(classpathStartIndex);

            Config config = ConfigFactory.parseString(
                    LiveResourceFactory.get(relativePath).getContent()
            );
            var ppcDataExportInitInfo = new PpcDataExportInitInfo(config, relativePath, defaultYtClusters);

            if (isEnvironmentSuitable(ppcDataExportInitInfo.getEnvironmentCondition())) {
                result.add(ppcDataExportInitInfo);
            }
        }
        return result;
    }

    private boolean isEnvironmentSuitable(Class<EnvironmentCondition> envCondition) {
        try {
            return context
                    .getBean(envCondition)
                    .evaluate();
        } catch (BeansException e) {
            logger.warn("Cannot create EnvironmentCondition bean from: " + envCondition, e);
            return false;
        }
    }

}
