package ru.yandex.direct.jobs.configuration;

import com.yandex.ydb.core.auth.AuthProvider;
import com.yandex.ydb.core.grpc.GrpcTransport;
import com.yandex.ydb.core.rpc.RpcTransport;
import com.yandex.ydb.table.SessionRetryContext;
import com.yandex.ydb.table.TableClient;
import com.yandex.ydb.table.rpc.grpc.GrpcTableRpc;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import org.springframework.context.annotation.Lazy;

import ru.yandex.direct.common.configuration.TvmIntegrationConfiguration;
import ru.yandex.direct.tvm.TvmIntegration;
import ru.yandex.direct.ydb.YdbPath;
import ru.yandex.direct.ydb.client.YdbSessionProperties;
import ru.yandex.direct.ydb.tvm.YdbTvmAuthProvider;

@Lazy
@Configuration
@Import(TvmIntegrationConfiguration.class)
public class MaintenanceHelpersYdbConfiguration {

    public static final String MAINTENANCE_HELPERS_YDB_SESSION_PROPERTIES_BEAN =
            "MaintenanceHelpersYdbSessionProperties";
    public static final String MAINTENANCE_HELPERS_YDB_PATH_BEAN = "MaintenanceHelpersYdbPath";
    public static final String MAINTENANCE_HELPERS_YDB_RPC_TRANSPORT_BEAN = "MaintenanceHelpersRpcTransport";
    public static final String MAINTENANCE_HELPERS_YDB_TABLE_CLIENT_BEAN = "MaintenanceHelpersTableClient";
    public static final String MAINTENANCE_HELPERS_YDB_SESSION_RETRY_CONTEXT_BEAN =
            "MaintenanceHelpersSessionRetryContext";

    @Bean(MAINTENANCE_HELPERS_YDB_PATH_BEAN)
    public YdbPath ydbPath(@Value("${ydb.cluster}") String cluster,
                           @Value("${ydb.maintenance_helpers.db_name}") String dbName) {
        return YdbPath.of(cluster, dbName);
    }

    @Bean
    public AuthProvider authProvider(TvmIntegration tvmIntegration) {
        return new YdbTvmAuthProvider(tvmIntegration);
    }

    @Bean(MAINTENANCE_HELPERS_YDB_RPC_TRANSPORT_BEAN)
    public RpcTransport grpcTransport(
            @Value("${ydb.endpoint}") String endpoint,
            @Qualifier(MAINTENANCE_HELPERS_YDB_PATH_BEAN) YdbPath dbPath,
            AuthProvider authProvider
    ) {
        GrpcTransport.Builder transportBuilder = GrpcTransport.forEndpoint(endpoint, dbPath.getPath())
                .withAuthProvider(authProvider);
        return transportBuilder.build();
    }

    @Bean(MAINTENANCE_HELPERS_YDB_SESSION_PROPERTIES_BEAN)
    public YdbSessionProperties ydbProperties(
            @Value("${ydb.maintenance_helpers.max_query_retries}") int maxQueryRetries,
            @Value("${ydb.maintenance_helpers.retry_not_found}") boolean retryNotFound
    ) {
        return YdbSessionProperties.builder()
                .withMaxQueryRetries(maxQueryRetries)
                .withRetryNotFound(retryNotFound)
                .build();
    }

    @Bean(MAINTENANCE_HELPERS_YDB_TABLE_CLIENT_BEAN)
    public TableClient tableClient(
            @Qualifier(MAINTENANCE_HELPERS_YDB_RPC_TRANSPORT_BEAN) RpcTransport rpcTransport,
            @Value("${ydb.maintenance_helpers.session_pool_min_size}") int sessionPoolMinSize,
            @Value("${ydb.maintenance_helpers.session_pool_max_size}") int sessionPoolMaxSize) {
        return TableClient.newClient(GrpcTableRpc.useTransport(rpcTransport))
                .sessionPoolSize(sessionPoolMinSize, sessionPoolMaxSize)
                .build();
    }

    @Bean(MAINTENANCE_HELPERS_YDB_SESSION_RETRY_CONTEXT_BEAN)
    public SessionRetryContext sessionRetryContext(
            @Qualifier(MAINTENANCE_HELPERS_YDB_TABLE_CLIENT_BEAN) TableClient tableClient,
            @Qualifier(MAINTENANCE_HELPERS_YDB_SESSION_PROPERTIES_BEAN) YdbSessionProperties ydbSessionProperties) {
        return SessionRetryContext.create(tableClient)
                .maxRetries(ydbSessionProperties.getMaxQueryRetries())
                .retryNotFound(ydbSessionProperties.isRetryNotFound())
                .build();
    }
}
