package ru.yandex.direct.jobs.configuration;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.scheduling.TaskScheduler;

import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.core.entity.moderation.model.ModerationErrorToken;
import ru.yandex.direct.core.entity.moderation.service.ModerationObjectType;
import ru.yandex.direct.env.EnvironmentType;
import ru.yandex.direct.ess.common.logbroker.LogbrokerClientFactoryFacade;
import ru.yandex.direct.jobs.moderation.config.DestConfig;
import ru.yandex.direct.jobs.moderation.config.ErrorLogbrokerConsumerPropertiesHolder;
import ru.yandex.direct.jobs.moderation.config.ResponseModerationLogbrokerConsumerPropertiesHolder;
import ru.yandex.direct.jobs.moderation.config.ResponseModerationParameters;
import ru.yandex.direct.jobs.moderation.config.ResponseRoutingLogbrokerConsumerPropertiesHolder;
import ru.yandex.direct.jobs.moderation.config.ResponseRoutingParameters;
import ru.yandex.direct.jobs.moderation.config.UnparsedLogbrokerConsumerPropertiesHolder;
import ru.yandex.direct.jobs.moderation.errors.ModerationErrorCodeToJugglerStatus;
import ru.yandex.direct.jobs.moderation.processor.ModerationResponseProcessorFilter;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.tvm.TvmIntegration;
import ru.yandex.direct.tvm.TvmIntegrationImpl;
import ru.yandex.direct.tvm.TvmIntegrationStub;
import ru.yandex.direct.tvm.TvmService;
import ru.yandex.kikimr.persqueue.auth.Credentials;
import ru.yandex.kikimr.persqueue.compression.CompressionCodec;

import static com.google.common.base.Preconditions.checkNotNull;
import static ru.yandex.direct.config.EssentialConfiguration.CONFIG_SCHEDULER_BEAN_NAME;

public class ModerationConfiguration {

    public static final String MODERATION_CONSUMER_BEAN = "moderation_consumer";
    public static final String MODERATION_ROUTER_TVM_INTEGRATION_BEAN_NAME = "moderationRouterTvmIntegration";
    public static final String MODERATION_ROUTER_LOGBROKER_CLIENT_FACTORY_BEAN_NAME =
            "moderationRouterLogbrokerClientFactory";

    @Bean(MODERATION_CONSUMER_BEAN)
    public String consumer(@Value("${moderation_service.logbroker.consumer}") String consumer) {
        return consumer;
    }

    @Bean
    public ResponseModerationLogbrokerConsumerPropertiesHolder responseModerationLogbrokerReaderConfig(
            DirectConfig config,
            @Value("${moderation_service.logbroker.host}") String logbrokerHost,
            @Qualifier(MODERATION_CONSUMER_BEAN) String consumerName,
            @Value("${moderation_service.logbroker.data_timeout}") int logbrokerDataTimeoutSec,
            @Value("${moderation_service.logbroker.init_timeout}") int logbrokerInitTimeoutSec,
            @Value("${moderation_service.logbroker.retries}") int retries
    ) {
        List<DirectConfig> topicsConfigs = config.getConfigList("moderation_service.logbroker.response.topics");
        return ResponseModerationLogbrokerConsumerPropertiesHolder.builder()
                .withConsumerName(consumerName)
                .withHost(logbrokerHost)
                .withLogbrokerDataTimeoutSec(logbrokerDataTimeoutSec)
                .withInitTimeoutSec(logbrokerInitTimeoutSec)
                .withRetries(retries)
                .withReadDataAfterTimestampMs(null)
                .withReadTopicsGroups(parseReadTopicsGroups(topicsConfigs))
                .build();
    }

    @Bean
    public ErrorLogbrokerConsumerPropertiesHolder errorResponseModerationLogbrokerReaderConfig(
            DirectConfig directConfig,
            @Value("${moderation_service.logbroker.host}") String logbrokerHost,
            @Qualifier(MODERATION_CONSUMER_BEAN) String consumerName,
            @Value("${moderation_service.logbroker.data_timeout}") int logbrokerDataTimeoutSec,
            @Value("${moderation_service.logbroker.init_timeout}") int logbrokerInitTimeoutSec,
            @Value("${moderation_service.logbroker.retries}") int retries,
            @Value("${moderation_service.logbroker.error.topic}") String logbrokerTopic) {

        var logbrokerConf = directConfig.getBranch("moderation_service.logbroker");

        return ErrorLogbrokerConsumerPropertiesHolder.builder()
                .withConsumerName(consumerName)
                .withHost(logbrokerHost)
                .withLogbrokerDataTimeoutSec(logbrokerDataTimeoutSec)
                .withInitTimeoutSec(logbrokerInitTimeoutSec)
                .withRetries(retries)
                .withGroups(logbrokerConf.getIntList("error.group"))
                .withReadDataAfterTimestampMs(null)
                .withReadTopic(logbrokerTopic)
                .build();
    }

    @Bean
    public ModerationErrorCodeToJugglerStatus moderationErrorCodeToJugglerStatus(DirectConfig config) {
        Map<String, String> errorSetting =
                config.getBranch("moderation_service.logbroker.error.code_levels").asMap();

        Map<ModerationErrorToken, JugglerStatus> converted = new HashMap<>();

        for (String strCode : errorSetting.keySet()) {
            ModerationErrorToken code = ModerationErrorToken.valueOf(strCode.toUpperCase());
            JugglerStatus jugglerStatus = JugglerStatus.valueOf(errorSetting.get(strCode).toUpperCase());
            converted.put(code, jugglerStatus);
        }

        return new ModerationErrorCodeToJugglerStatus(converted);
    }

    @Bean
    public UnparsedLogbrokerConsumerPropertiesHolder unparsedResponseModerationLogbrokerReaderConfig(
            DirectConfig directConfig,
            @Value("${moderation_service.logbroker.host}") String logbrokerHost,
            @Qualifier(MODERATION_CONSUMER_BEAN) String consumerName,
            @Value("${moderation_service.logbroker.data_timeout}") int logbrokerDataTimeoutSec,
            @Value("${moderation_service.logbroker.init_timeout}") int logbrokerInitTimeoutSec,
            @Value("${moderation_service.logbroker.retries}") int retries,
            @Value("${moderation_service.logbroker.unparsed.topic}") String logbrokerTopic) {

        var logbrokerConf = directConfig.getBranch("moderation_service.logbroker");

        return UnparsedLogbrokerConsumerPropertiesHolder.builder()
                .withConsumerName(consumerName)
                .withHost(logbrokerHost)
                .withLogbrokerDataTimeoutSec(logbrokerDataTimeoutSec)
                .withInitTimeoutSec(logbrokerInitTimeoutSec)
                .withRetries(retries)
                .withGroups(logbrokerConf.getIntList("unparsed.group"))
                .withReadDataAfterTimestampMs(null)
                .withReadTopic(logbrokerTopic)
                .build();
    }

    @Bean
    public ResponseModerationParameters responseModerationParameters() {
        return new ResponseModerationParameters.Builder()
                .setLogbrokerNoCommit(false)
                .build();
    }

    @Bean
    public ModerationResponseProcessorFilter moderationResponseProcessorFilter(EnvironmentType environmentType) {
        if (!environmentType.isBeta()) {
            return ModerationResponseProcessorFilter.doNothing();
        }

        return new ModerationResponseProcessorFilter(r -> r.getAttributes() == null || r.getAttributes().getBetaPort() == null);
    }

    @Bean(name = MODERATION_ROUTER_TVM_INTEGRATION_BEAN_NAME)
    public TvmIntegration moderationTvmIntegration(DirectConfig directConfig,
                                                   @Qualifier(CONFIG_SCHEDULER_BEAN_NAME) TaskScheduler liveConfigChangeTaskScheduler) {
        boolean enabled = directConfig.findBoolean("moderation_service.tvm.enabled").orElse(true);
        if (enabled) {
            return TvmIntegrationImpl.create(directConfig.getBranch("moderation_service"),
                    liveConfigChangeTaskScheduler);
        } else {
            return new TvmIntegrationStub();
        }
    }

    @Bean(name = MODERATION_ROUTER_LOGBROKER_CLIENT_FACTORY_BEAN_NAME)
    public LogbrokerClientFactoryFacade logbrokerClientCreatorFactory(
            DirectConfig directConfig,
            @Qualifier(MODERATION_ROUTER_TVM_INTEGRATION_BEAN_NAME) TvmIntegration tvmIntegration) {
        TvmService tvmService = TvmService.fromStringStrict(
                directConfig.getString("moderation_service.logbroker.tvm_service_name"));
        return new LogbrokerClientFactoryFacade(() -> {
            String serviceTicket = tvmIntegration.getTicket(tvmService);
            return Credentials.tvm(serviceTicket);
        });
    }

    @Bean
    public ResponseRoutingParameters responseRoutingParameters(
            DirectConfig config,
            @Value("${moderation_service.logbroker.host}") String logbrokerHost,
            @Value("${moderation_service.logbroker.data_timeout}") int logbrokerDataTimeoutSec,
            @Value("${moderation_service.logbroker.init_timeout}") int logbrokerInitTimeoutSec,
            @Value("${moderation_service.logbroker.retries}") int retries,
            @Value("${moderation_service.logbroker.routing.consumer}") String consumerName,
            @Value("${moderation_service.logbroker.routing.read_topic}") String readTopic,
            @Value("${moderation_service.logbroker.routing.default_write_topic}") String defaultWriteTopic,
            @Value("${moderation_service.logbroker.routing.default_write_groups}") int defaultWriteGroups
    ) {
        ResponseRoutingLogbrokerConsumerPropertiesHolder consumerPropertiesHolder =
                ResponseRoutingLogbrokerConsumerPropertiesHolder.builder()
                        .withConsumerName(consumerName)
                        .withHost(logbrokerHost)
                        .withLogbrokerDataTimeoutSec(logbrokerDataTimeoutSec)
                        .withInitTimeoutSec(logbrokerInitTimeoutSec)
                        .withRetries(retries)
                        .withGroups(config.getIntList("moderation_service.logbroker.routing.read_groups"))
                        .withReadDataAfterTimestampMs(null)
                        .withReadTopic(readTopic)
                        .build();
        DirectConfig logbrokerConfig = config.getBranch("moderation_service.logbroker");
        CompressionCodec compressionCodec =
                CompressionCodec.valueOf(logbrokerConfig.findString("compression_codec").orElse("gzip").toUpperCase());
        ResponseRoutingParameters.ProducerProperties producerProperties =
                new ResponseRoutingParameters.ProducerProperties(
                        logbrokerHost,
                        logbrokerDataTimeoutSec,
                        retries,
                        compressionCodec
                );
        List<DirectConfig> routesConfigs = config.getConfigList("moderation_service.logbroker.routing.routes");
        Map<ModerationObjectType, DestConfig> routesMap = StreamEx.of(routesConfigs)
                .toMap(routeConfig -> {
                            var typeName = routeConfig.getString("type");
                            var type = ModerationObjectType.getEnumByValue(typeName);
                            checkNotNull(type, "can't find enum for type: " + typeName);
                            return type;
                        },
                        routeConfig -> {
                            String writeTopic = routeConfig.getString("write_topic");
                            int writeGroups = routeConfig.getInt("write_groups");
                            return new DestConfig(writeTopic, writeGroups);
                        });
        DestConfig defaultDestConfig = new DestConfig(defaultWriteTopic, defaultWriteGroups);
        return new ResponseRoutingParameters(consumerPropertiesHolder, producerProperties, defaultDestConfig, routesMap);
    }

    @Bean
    public ResponseRoutingLogbrokerConsumerPropertiesHolder responseRoutingLogbrokerConsumerPropertiesHolder(
            ResponseRoutingParameters responseRoutingParameters
    ) {
        return responseRoutingParameters.getConsumerProperties();
    }

    public static Map<String, List<Integer>> parseReadTopicsGroups(List<DirectConfig> topicsConfigs) {
        return StreamEx.of(topicsConfigs)
                .mapToEntry(cfg -> cfg.getString("topic"), cfg -> cfg.getIntList("group"))
                .toMap();
    }
}
