package ru.yandex.direct.jobs.configuration;

import java.time.Duration;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.scheduling.TaskScheduler;

import ru.yandex.direct.cloud.iam.IIamTokenProvider;
import ru.yandex.direct.cloud.iam.service.CloudIamTokenProviderService;
import ru.yandex.direct.cloud.iam.service.DummyIamTokenProvider;
import ru.yandex.direct.cloud.mdb.mysql.api.service.CloudMdbMySqlApiService;
import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.env.EnvironmentType;
import ru.yandex.direct.ess.common.logbroker.LogbrokerClientFactoryFacade;
import ru.yandex.direct.jobs.slowlogs.MySqlSlowQueryLogsWriterJobPropertiesHolder;
import ru.yandex.direct.jobs.slowlogs.MySqlSlowQueryLogsWriterLogbrokerPropertiesHolder;
import ru.yandex.direct.liveresource.LiveResource;
import ru.yandex.direct.liveresource.LiveResourceFactory;
import ru.yandex.direct.mysql.slowlog.writer.states.MySqlClusterSlowLogsWriterStateProvider;
import ru.yandex.direct.tvm.TvmIntegration;
import ru.yandex.direct.tvm.TvmService;
import ru.yandex.kikimr.persqueue.auth.Credentials;
import ru.yandex.kikimr.persqueue.compression.CompressionCodec;

import static ru.yandex.direct.common.configuration.TvmIntegrationConfiguration.TVM_INTEGRATION;
import static ru.yandex.direct.config.EssentialConfiguration.CONFIG_SCHEDULER_BEAN_NAME;

@Configuration
public class MySqlSlowQueryLogsConfiguration {
    private static final Logger logger = LoggerFactory.getLogger(MySqlSlowQueryLogsConfiguration.class);

    public static final String MYSQL_SLOW_QUERY_LOGS_WRITER_CONFIG_SECTION_NAME = "cloud_mdb_mysql_slow_query_logs";
    public static final String DEFAULT_CLOUD_IAM_TOKEN_PROVIDER_BEAN_NAME = "iamTokenProvider";
    public static final String DEFAULT_CLOUD_MDB_MYSQL_API_BEAN_NAME = "cloudMdbMySqlApi";
    public static final String MYSQL_SLOW_QUERY_LOGS_WRITER_LOGBROKER_CLIENT_FACTORY_BEAN_NAME =
            "mySqlSlowQueryLogsWriterLogbrokerClientFactory";

    @Bean(name = DEFAULT_CLOUD_IAM_TOKEN_PROVIDER_BEAN_NAME)
    public IIamTokenProvider iamTokenProvider(
            DirectConfig directConfig,
            EnvironmentType environmentType,
            @Qualifier(CONFIG_SCHEDULER_BEAN_NAME) TaskScheduler liveConfigChangeTaskScheduler) {
        DirectConfig cloudIamProperties = directConfig.getBranch("cloud_iam_default");
        boolean enabled = cloudIamProperties.findBoolean("enabled").orElse(false);
        if (enabled) {
            if (environmentType.isDevelopment()) {
                String oAuthTokenUrl = cloudIamProperties.getString("oauth_token_provider.oauth_token_file");
                String content = null;
                Exception inner = null;
                try {
                    LiveResource oauthTokenResource = LiveResourceFactory.get(oAuthTokenUrl);
                    content = oauthTokenResource.getContent();
                } catch (Exception ex) {
                    inner = ex;
                }
                if (content == null || content.isEmpty()) {
                    return new DummyIamTokenProvider(String.format("Resource with cloud oauth token '%s' " +
                            "was not found or it cannot be read, or its content is empty. " +
                            "Unable to create real cloud iam token provider, dummy stub was created instead. " +
                            "Please check cloud_iam_default.oauth_token_provider.oauth_token_file parameter " +
                            "in config files", oAuthTokenUrl), inner);
                }
            }
            return CloudIamTokenProviderService.create(
                    directConfig.getBranch("cloud"),
                    cloudIamProperties.getBranch("oauth_token_provider"),
                    liveConfigChangeTaskScheduler);
        } else {
            return new DummyIamTokenProvider("Iam token provider is disabled in config. " +
                    "Please check cloud_iam_default.enabled parameter in config files");
        }
    }

    @Bean(name = DEFAULT_CLOUD_MDB_MYSQL_API_BEAN_NAME)
    public CloudMdbMySqlApiService cloudMdbMySqlApi(
            DirectConfig directConfig,
            @Qualifier(DEFAULT_CLOUD_IAM_TOKEN_PROVIDER_BEAN_NAME) IIamTokenProvider iamTokenProvider) {
        return CloudMdbMySqlApiService.create(directConfig.getBranch("cloud"), iamTokenProvider);
    }

    @Bean
    public MySqlClusterSlowLogsWriterStateProvider mySqlClusterSlowLogsWriterStateProvider(
            PpcPropertiesSupport ppcPropertiesSupport) {
        return new MySqlClusterSlowLogsWriterStateProvider(ppcPropertiesSupport);
    }

    @Bean
    public MySqlClustersParametersSource mySqlClustersParametersSource(
            DirectConfig directConfig) {
        List<String> clustersNames = directConfig.getBranch(MYSQL_SLOW_QUERY_LOGS_WRITER_CONFIG_SECTION_NAME)
                .getStringList("clusters");
        return new MySqlClustersParametersSource(clustersNames);
    }

    @Bean
    public MySqlSlowQueryLogsWriterJobPropertiesHolder mySqlSlowQueryLogsWriterJobPropertiesHolder(
            DirectConfig directConfig) {
        DirectConfig jobProperties = directConfig.getBranch(MYSQL_SLOW_QUERY_LOGS_WRITER_CONFIG_SECTION_NAME)
                .getBranch("job");
        int cloudRowsInBatch = jobProperties.getInt("cloud_rows_count_in_batch");
        long rowsBufferSizeInBytes = jobProperties.getLong("log_rows_buffer_estimate_size_in_megabytes");
        rowsBufferSizeInBytes *= 1024 * 1024;
        String excludeQueriesFilter = jobProperties.getString("exclude_queries_filter");
        if (excludeQueriesFilter != null) {
            excludeQueriesFilter = excludeQueriesFilter.toLowerCase();
        }
        String directInfraCloudFolderId = jobProperties.getString("direct_infra_cloud_folder_id");
        int newClusterLogsStartDaysAgo = jobProperties.getInt("new_cluster_logs_start_days_ago");
        boolean enableSlowLogRecordRawTextField = jobProperties
                .findBoolean("enable_slow_log_record_raw_text_field").orElse(false);

        return new MySqlSlowQueryLogsWriterJobPropertiesHolder(
                cloudRowsInBatch, rowsBufferSizeInBytes, excludeQueriesFilter,
                newClusterLogsStartDaysAgo, directInfraCloudFolderId, enableSlowLogRecordRawTextField);
    }

    @Bean
    public MySqlSlowQueryLogsWriterLogbrokerPropertiesHolder mySqlSlowQueryLogsWriterLogbrokerPropertiesHolder(
            DirectConfig directConfig) {
        DirectConfig logbrokerProperties = directConfig.getBranch(MYSQL_SLOW_QUERY_LOGS_WRITER_CONFIG_SECTION_NAME)
                .getBranch("logbroker");
        String host = logbrokerProperties.getString("host");
        String topic = logbrokerProperties.getString("topic");
        String tvmServiceName = logbrokerProperties.getString("tvm_service_name");
        Duration timeout = logbrokerProperties.getDuration("timeout");
        CompressionCodec compressionCodec =
                CompressionCodec.valueOf(
                        logbrokerProperties.findString("compression_codec").orElse("raw").toUpperCase());
        return new MySqlSlowQueryLogsWriterLogbrokerPropertiesHolder(
                host, topic, tvmServiceName, timeout, compressionCodec);
    }

    @Bean(name = MYSQL_SLOW_QUERY_LOGS_WRITER_LOGBROKER_CLIENT_FACTORY_BEAN_NAME)
    public LogbrokerClientFactoryFacade logbrokerClientCreatorFactory(
            MySqlSlowQueryLogsWriterLogbrokerPropertiesHolder logbrokerProperties,
            @Qualifier(TVM_INTEGRATION) TvmIntegration tvmIntegration) {
        TvmService tvmService = TvmService.fromStringStrict(logbrokerProperties.getTvmServiceName());
        return new LogbrokerClientFactoryFacade(() -> {
            String serviceTicket = tvmIntegration.getTicket(tvmService);
            return Credentials.tvm(serviceTicket);
        });
    }
}
