package ru.yandex.direct.jobs.configuration;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.jobs.ppcdataexport.PpcDataExportJob;
import ru.yandex.direct.jobs.ppcdataexport.model.PpcDataExportInitInfo;
import ru.yandex.direct.scheduler.support.ParametersSource;
import ru.yandex.direct.ytwrapper.model.YtCluster;

import static com.google.common.base.Preconditions.checkArgument;
import static java.util.Collections.unmodifiableList;

/**
 * На каких кластерах и какие запросы выполнять в джобе {@link PpcDataExportJob}
 */
@ParametersAreNonnullByDefault
public class PpcDataExportParametersSource implements ParametersSource<PpcDataExportParameter> {
    private static final String SEPARATOR = "---";
    private static final int MAX_PATH_SIZE = 120;

    private final List<PpcDataExportParameter> parameters;

    public static PpcDataExportParametersSource from(List<PpcDataExportInitInfo> ppcDataExportInitInfoList) {
        checkPathSize(ppcDataExportInitInfoList);

        List<PpcDataExportParameter> parameters = new ArrayList<>();
        ppcDataExportInitInfoList.forEach(ppcDataExportInitInfo -> {
            String confFilePath = ppcDataExportInitInfo.getConfFilePath();
            Set<YtCluster> ytClusters = ppcDataExportInitInfo.getYtClusters();
            ytClusters.stream()
                    .map(ytCluster -> new PpcDataExportParameter(ytCluster, confFilePath))
                    .forEach(parameters::add);
        });

        return new PpcDataExportParametersSource(parameters);
    }

    private static void checkPathSize(List<PpcDataExportInitInfo> ppcDataExportInitInfoList) {
        ppcDataExportInitInfoList.stream()
                .map(PpcDataExportInitInfo::getConfFilePath)
                .forEach(p -> {
                    checkArgument(p.length() < MAX_PATH_SIZE, "file path %s must be shorter than %d", p, MAX_PATH_SIZE);
                    checkArgument(!p.contains(SEPARATOR), "file path %s must not contain separator %s", p, SEPARATOR);
                });
    }

    private PpcDataExportParametersSource(List<PpcDataExportParameter> parameters) {
        this.parameters = parameters;
    }

    @Override
    public List<PpcDataExportParameter> getAllParamValues() {
        return unmodifiableList(parameters);
    }

    @Override
    public String convertParamToString(PpcDataExportParameter paramValue) {
        return String.join(SEPARATOR,
                paramValue.getYtCluster().name(),
                paramValue.getConfFilePath()
        );
    }

    @Override
    public PpcDataExportParameter convertStringToParam(String string) {
        String[] parts = string.split(SEPARATOR);
        return new PpcDataExportParameter(
                YtCluster.valueOf(parts[0]),
                parts[1]
        );
    }
}
