package ru.yandex.direct.jobs.configuration;

import kotlin.Pair;
import kotlin.jvm.functions.Function1;
import one.util.streamex.StreamEx;
import org.springframework.context.annotation.Bean;

import ru.yandex.direct.jobs.monitoring.system.MonitoringSourceType;
import ru.yandex.direct.jobs.monitoring.system.processor.MonitoringDataProcessor;
import ru.yandex.direct.jobs.monitoring.system.processor.MultipleStringExistMonitoringProcessor;
import ru.yandex.direct.jobs.monitoring.system.processor.RateMonitoringProcessor;
import ru.yandex.direct.jobs.monitoring.system.processor.StringExistMonitoringProcessor;
import ru.yandex.direct.jobs.monitoring.system.processor.ThresholdMonitoringProcessor;
import ru.yandex.direct.jobs.monitoring.system.processor.TimelineMonitoringProcessor;
import ru.yandex.direct.jobs.monitoring.system.processor.TraceLogFunctionMonitoringProcessor;
import ru.yandex.direct.jobs.monitoring.system.source.MonitoringData;
import ru.yandex.direct.jobs.monitoring.system.source.tracelog.TraceLogFunctionRow;
import ru.yandex.direct.jobs.monitoring.system.source.tracelog.TraceLogMonitoringData;
import ru.yandex.monlib.metrics.histogram.Histograms;

import static ru.yandex.direct.jobs.monitoring.system.processor.ProcessorConstantsKt.EXTERNAL_SERVICES;
import static ru.yandex.direct.jobs.monitoring.system.processor.ProcessorConstantsKt.METRIKA_UNOPTIMIZED_REQUESTS;
import static ru.yandex.direct.jobs.monitoring.system.processor.ProcessorConstantsKt.MULTI_SHARD_THRESHOLD;
import static ru.yandex.direct.jobs.monitoring.system.processor.ProcessorConstantsKt.REAL_TIME_SERVICES;
import static ru.yandex.direct.jobs.monitoring.system.processor.ProcessorConstantsKt.YQL_FUNCTION_NAME;

public class SystemMonitoringConfiguration {

    private static final Integer DEFAULT_MULTIPLIER = 1000;
    private static final Integer BUCKETS_COUNT = 13;

    private static final String TRACE_LOG_RPS_MONITORING_PROCESSOR_NAME =
            "traceLogRPSMonitoringDataProcessor";
    private static final String TRACE_LOG_TIME_SPENT_MONITORING_PROCESSOR_NAME =
            "traceLogTimeSpentMonitoringDataProcessor";
    private static final String TRACE_LOG_TIME_SPENT_ACCURATE_MONITORING_PROCESSOR_NAME =
            "traceLogTimeSpentAccurateMonitoringDataProcessor";
    private static final String TRACE_LOG_TIME_SPENT_REST_MONITORING_PROCESSOR_NAME =
            "traceLogTimeSpentRestMonitoringDataProcessor";
    private static final String TRACE_LOG_CPU_MONITORING_PROCESSOR_NAME =
            "traceLogCpuMonitoringDataProcessor";
    private static final String TRACE_LOG_MEM_MONITORING_PROCESSOR_NAME =
            "traceLogMemMonitoringDataProcessor";
    private static final String TRACE_LOG_DB_READ_CALLS_MONITORING_PROCESSOR_NAME =
            "traceLogDbReadCallsMonitoringDataProcessor";
    private static final String TRACE_LOG_DB_WRITE_CALLS_MONITORING_PROCESSOR_NAME =
            "traceLogDbWriteCallsMonitoringDataProcessor";

    private static final String TRACE_LOG_METRIKA_UNOPTIMIZED_REQUESTS_MONITORING_DATA_PROCESSOR_NAME =
            "traceLogmetrikaUnoptimizedMonitoringDataProcessor";
    private static final String TRACE_LOG_MULTI_SHARD_MONITORING_DATA_PROCESSOR_NAME =
            "traceLogMultiShardMonitoringDataProcessor";
    private static final String TRACE_LOG_YQL_EXECUTION_MONITORING_DATA_PROCESSOR_NAME =
            "traceLogYqlExecuteMonitoringDataProcessor";

    private static final String TRACE_LOG_EXTERNAL_SERVICE_MONITORING_DATA_PROCESSOR_NAME =
            "traceLogExternalServiceMonitoringDataProcessor";

    private static final String TRACE_LOG_FUNCTION_SPEND_TIME_MONITORING_PROCESSOR_NAME =
            "traceLogFunctionSpendTimeMonitoringDataProcessor";
    private static final String TRACE_LOG_FUNCTION_CALLS_MONITORING_PROCESSOR_NAME =
            "traceLogFunctionCallsMonitoringDataProcessor";
    private static final String TRACE_LOG_FUNCTION_OBJECT_NUMBER_MONITORING_PROCESSOR_NAME =
            "traceLogFunctionObjectNumberMonitoringDataProcessor";

    //processors

    //вспомогательный процессор для определения частотности выполнения запросов.
    //Используется для фильтрации в мониторинги редких запросов
    @Bean(TRACE_LOG_RPS_MONITORING_PROCESSOR_NAME)
    public MonitoringDataProcessor<? extends MonitoringData> traceLogRPSMonitoringDataProcessor() {
        return new RateMonitoringProcessor<>(
                MonitoringSourceType.TRACE_LOG,
                REAL_TIME_SERVICES
        );
    }


    @Bean(TRACE_LOG_TIME_SPENT_MONITORING_PROCESSOR_NAME)
    public MonitoringDataProcessor<? extends MonitoringData> traceLogTimeSpentMonitoringDataProcessor() {
        return new TimelineMonitoringProcessor<>(
                MonitoringSourceType.TRACE_LOG,
                REAL_TIME_SERVICES,
                TraceLogMonitoringData::getTimeSpent,
                DEFAULT_MULTIPLIER,
                () -> Histograms.exponential(BUCKETS_COUNT, 1.5, 400.0)
        );
    }

    @Bean(TRACE_LOG_TIME_SPENT_ACCURATE_MONITORING_PROCESSOR_NAME)
    public MonitoringDataProcessor<? extends MonitoringData> traceLogTimeSpentAccurateMonitoringDataProcessor() {
        return new TimelineMonitoringProcessor<>(
                MonitoringSourceType.TRACE_LOG,
                REAL_TIME_SERVICES,
                TraceLogMonitoringData::getTimeSpent,
                DEFAULT_MULTIPLIER,
                () -> Histograms.exponential(24, 1.3, 100.0)
        );
    }

    @Bean(TRACE_LOG_FUNCTION_SPEND_TIME_MONITORING_PROCESSOR_NAME)
    public MonitoringDataProcessor<? extends MonitoringData> traceLogFunctionSpendTimeMonitoringDataProcessor() {
        return new TraceLogFunctionMonitoringProcessor<>(
                MonitoringSourceType.TRACE_LOG,
                "all_ela",
                (functionRow, oldValue) -> functionRow.getEla() + oldValue,
                0d,
                1000
        );
    }

    @Bean(TRACE_LOG_FUNCTION_CALLS_MONITORING_PROCESSOR_NAME)
    public MonitoringDataProcessor<? extends MonitoringData> traceLogFunctionCallsMonitoringDataProcessor() {
        return new TraceLogFunctionMonitoringProcessor<>(
                MonitoringSourceType.TRACE_LOG,
                "calls",
                (functionRow, oldValue) -> functionRow.getCalls() + oldValue,
                0L
        );
    }

    @Bean(TRACE_LOG_FUNCTION_OBJECT_NUMBER_MONITORING_PROCESSOR_NAME)
    public MonitoringDataProcessor<? extends MonitoringData> traceLogFunctionObjectNumberMonitoringDataProcessor() {
        return new TraceLogFunctionMonitoringProcessor<>(
                MonitoringSourceType.TRACE_LOG,
                "obj_num",
                (functionRow, oldValue) -> functionRow.getObjectNum() + oldValue,
                0L
        );
    }

    @Bean(TRACE_LOG_TIME_SPENT_REST_MONITORING_PROCESSOR_NAME)
    public MonitoringDataProcessor<? extends MonitoringData> traceLogTimeSpentRestMonitoringDataProcessor() {
        return new TimelineMonitoringProcessor<TraceLogMonitoringData>(
                MonitoringSourceType.TRACE_LOG,
                REAL_TIME_SERVICES,
                r -> {
                    var rest = r.getFunctionsProfile()
                            .stream().filter(f -> f.getName().equals("rest")).findFirst();
                    return rest.isEmpty() ? 0 : rest.get().getEla();
                },
                DEFAULT_MULTIPLIER,
                () -> Histograms.exponential(9, 1.5, 300.0)
        );
    }

    @Bean(TRACE_LOG_CPU_MONITORING_PROCESSOR_NAME)
    public MonitoringDataProcessor<? extends MonitoringData> traceLogCpuMonitoringDataProcessor() {
        return new TimelineMonitoringProcessor<>(
                MonitoringSourceType.TRACE_LOG,
                REAL_TIME_SERVICES,
                TraceLogMonitoringData::getCpu,
                10,
                () -> Histograms.exponential(BUCKETS_COUNT, 2, 1.0)
        );
    }

    @Bean(TRACE_LOG_MEM_MONITORING_PROCESSOR_NAME)
    public MonitoringDataProcessor<? extends MonitoringData> traceLogMemMonitoringDataProcessor() {
        return new TimelineMonitoringProcessor<>(
                MonitoringSourceType.TRACE_LOG,
                REAL_TIME_SERVICES,
                TraceLogMonitoringData::getMemory,
                null,
                () -> Histograms.exponential(11, 1.7, 100.0)
        );
    }

    @Bean(TRACE_LOG_DB_READ_CALLS_MONITORING_PROCESSOR_NAME)
    public MonitoringDataProcessor<? extends MonitoringData> traceLogDbReadCallsMonitoringDataProcessor() {
        return new TimelineMonitoringProcessor<>(
                MonitoringSourceType.TRACE_LOG,
                REAL_TIME_SERVICES,
                getDbReadCallsExtractor(),
                null,
                () -> Histograms.exponential(11, 1.5, 5)
        );
    }

    @Bean(TRACE_LOG_DB_WRITE_CALLS_MONITORING_PROCESSOR_NAME)
    public MonitoringDataProcessor<? extends MonitoringData> traceLogDbWriteCallsMonitoringDataProcessor() {
        return new TimelineMonitoringProcessor<>(
                MonitoringSourceType.TRACE_LOG,
                REAL_TIME_SERVICES,
                getDbWriteCallsExtractor(),
                null,
                () -> Histograms.exponential(11, 1.5, 5)
        );
    }

    @Bean(TRACE_LOG_MULTI_SHARD_MONITORING_DATA_PROCESSOR_NAME)
    public MonitoringDataProcessor<? extends MonitoringData> traceLogMultiShardMonitoringDataProcessor(
    ) {
        return new ThresholdMonitoringProcessor<>(
                MonitoringSourceType.TRACE_LOG,
                REAL_TIME_SERVICES,
                MULTI_SHARD_THRESHOLD,
                getMultiShardExtractor()
        );
    }

    @Bean(TRACE_LOG_YQL_EXECUTION_MONITORING_DATA_PROCESSOR_NAME)
    public MonitoringDataProcessor<? extends MonitoringData> traceLogYqlExecuteMonitoringDataProcessor(
    ) {
        return new StringExistMonitoringProcessor<TraceLogMonitoringData>(
                MonitoringSourceType.TRACE_LOG,
                REAL_TIME_SERVICES,
                YQL_FUNCTION_NAME,
                dataRow -> StreamEx.of(dataRow.getFunctionsProfile())
                        .map(TraceLogFunctionRow::getName)
                        .toSet()
        );
    }

    @Bean(TRACE_LOG_EXTERNAL_SERVICE_MONITORING_DATA_PROCESSOR_NAME)
    public MonitoringDataProcessor<? extends MonitoringData> traceLogExternalServiceMonitoringDataProcessor() {
        return new MultipleStringExistMonitoringProcessor<TraceLogMonitoringData>(
                MonitoringSourceType.TRACE_LOG,
                REAL_TIME_SERVICES,
                EXTERNAL_SERVICES,
                false,
                dataRow -> StreamEx.of(dataRow.getFunctionsProfile())
                        .map(r -> new Pair<>(r.getName(), r.getTags()))
                        .toList()
        );
    }

    @Bean(TRACE_LOG_METRIKA_UNOPTIMIZED_REQUESTS_MONITORING_DATA_PROCESSOR_NAME)
    public MonitoringDataProcessor<? extends MonitoringData> traceLogmetrikaUnoptimizedMonitoringDataProcessor(
    ) {
        return new MultipleStringExistMonitoringProcessor<TraceLogMonitoringData>(
                MonitoringSourceType.TRACE_LOG,
                REAL_TIME_SERVICES,
                METRIKA_UNOPTIMIZED_REQUESTS,
                true,
                dataRow -> StreamEx.of(dataRow.getFunctionsProfile())
                        .map(r -> new Pair<>(r.getName(), r.getTags()))
                        .toList()
        );
    }

    private Function1<TraceLogMonitoringData, Double> getDbReadCallsExtractor() {
        return r -> {
            var dbRead = StreamEx.of(r.getFunctionsProfile())
                    .filter(f -> "db:read".equals(f.getName()))
                    .map(TraceLogFunctionRow::getCalls)
                    .reduce(Integer::sum);
            return dbRead.isEmpty() ? 0 : (double) dbRead.get();
        };
    }

    private Function1<TraceLogMonitoringData, Double> getDbWriteCallsExtractor() {
        return r -> {
            var dbWrite = StreamEx.of(r.getFunctionsProfile())
                    .filter(f -> "db:write".equals(f.getName()))
                    .map(TraceLogFunctionRow::getCalls)
                    .reduce(Integer::sum);
            return dbWrite.isEmpty() ? 0 : (double) dbWrite.get();
        };
    }

    private Function1<TraceLogMonitoringData, Integer> getMultiShardExtractor() {
        return r -> {
            var shardsWrite = StreamEx.of(r.getFunctionsProfile())
                    .filter(f -> "db:write".equals(f.getName()))
                    .map(TraceLogFunctionRow::getTags);
            var allShards = StreamEx.of(r.getFunctionsProfile())
                    .filter(f -> "db:read".equals(f.getName()))
                    .map(TraceLogFunctionRow::getTags)
                    .append(shardsWrite)
                    .nonNull()
                    .toSet();
            return allShards.size();
        };
    }
}
