package ru.yandex.direct.jobs.configuration;

import java.util.regex.Pattern;

import javax.annotation.Nullable;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;

import ru.yandex.direct.common.jetty.JettyLauncher;
import ru.yandex.direct.scheduler.SpringAppTerminator;

import static com.google.common.base.Preconditions.checkNotNull;

/**
 * Инициирует завершение приложения в случае, если 'наша' версия {@link #oursVersion} не совпадает с проверяемой версией
 *
 * @see SpringAppTerminator#terminate(int). Во внимание принимается только первая строка версии.
 * Версии проверяются на точное соответствие; как следствие, к завершению приложения приведет как
 * downgrade, так и upgrade.
 */
@Component
public class VersionMismatchTerminator {
    private static final Logger logger = LoggerFactory.getLogger(VersionMismatchTerminator.class);
    private static final Pattern END_OF_LINE_PATTERN = Pattern.compile("\r\n|\r|\n");
    private final SpringAppTerminator appTerminator;
    private final String oursVersion;
    private final JettyLauncher monitoringHttpServer;

    @Autowired
    public VersionMismatchTerminator(@Value("${job-scheduler-version:0.0-0}") String oursVersion,
                                     SpringAppTerminator appTerminator,
                                     @Nullable @Qualifier("monitoringHttpServer") JettyLauncher monitoringHttpServer) {
        this.appTerminator = checkNotNull(appTerminator);
        this.oursVersion = extractVersionString(checkNotNull(oursVersion));
        this.monitoringHttpServer = monitoringHttpServer;
    }

    public void terminateIfVersionMismatch(String newVersion) {
        String sanitizedNewVersion = extractVersionString(newVersion);
        if (!oursVersion.equals(sanitizedNewVersion)) {
            logger.error("version mismatch detected, terminating app(ours version: {}, new version: {})", oursVersion,
                    sanitizedNewVersion);
            if (monitoringHttpServer != null) {
                try {
                    monitoringHttpServer.stop();
                } catch (RuntimeException e) {
                    logger.error("Can't stop monitoring server", e);
                }
            }
            appTerminator.terminate(0);
        }
    }

    private static String extractVersionString(String srcString) {
        String[] lines = END_OF_LINE_PATTERN.split(srcString);
        if (lines.length > 1) {
            logger.warn("version contains more than 1 line; using first line as version");
        }
        return lines[0];
    }
}
