package ru.yandex.direct.jobs.contentcategories;

import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod;
import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.env.TypicalEnvironment;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification;
import ru.yandex.direct.juggler.check.model.CheckTag;
import ru.yandex.direct.juggler.check.model.NotificationRecipient;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectJob;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.direct.ytwrapper.model.YtSQLSyntaxVersion;
import ru.yandex.inside.yt.kosher.cypress.Cypress;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;

import static java.time.temporal.ChronoUnit.DAYS;
import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.jobs.contentcategories.ContentCategoriesService.MODIFICATION_TIME_ATTRIBUTE;
import static ru.yandex.direct.jobs.contentcategories.ContentCategoriesService.TYPE_ATTRIBUTE;
import static ru.yandex.direct.jobs.contentcategories.ContentCategoriesService.isTableNode;
import static ru.yandex.direct.jobs.contentcategories.ContentCategoriesService.parseModificationTimestamp;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_2;

/**
 * Джоба удаления старых таблиц для разметки url-ов категориями контента
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(days = 2),
        tags = {DIRECT_PRIORITY_2, CheckTag.DIRECT_PRODUCT_TEAM, CheckTag.YT},
        notifications = {
                @OnChangeNotification(recipient = NotificationRecipient.LOGIN_IVATKOEGOR,
                        status = {JugglerStatus.OK, JugglerStatus.CRIT},
                        method = NotificationMethod.TELEGRAM),
        },
        needCheck = ProductionOnly.class
)
@Hourglass(cronExpression = "0 0 0 * * ?", needSchedule = TypicalEnvironment.class)
@ParametersAreNonnullByDefault
public class ContentCategoriesOldTablesCleanerJob extends DirectJob {
    private static final Logger logger = LoggerFactory.getLogger(ContentCategoriesOldTablesCleanerJob.class);

    private static final YtCluster YT_CLUSTER = YtCluster.HAHN;

    private final DirectConfig contentCategoriesConfig;
    private final YtProvider ytProvider;

    @Autowired
    public ContentCategoriesOldTablesCleanerJob(
            DirectConfig directConfig,
            YtProvider ytProvider
    ) {
        this.contentCategoriesConfig = directConfig.getBranch("content_categories");
        this.ytProvider = ytProvider;
    }

    @Override
    public void execute() {
        var ytOperator = ytProvider.getOperator(YT_CLUSTER, YtSQLSyntaxVersion.SQLv1);
        var cypress = ytOperator.getYt().cypress();

        var cleanerConfig = contentCategoriesConfig.getBranch("tables_cleaner");
        List<DirectConfig> configs = cleanerConfig.getConfigList("configs");

        for (var config : configs) {
            var lifeTime = config.getLong("life_time_in_days");
            var thresholdTime = LocalDateTime.now().minus(lifeTime, DAYS).toEpochSecond(ZoneOffset.UTC);
            var maxCount = config.getLong("max_count");

            for (var folderPath : config.getStringList("folder_paths")) {
                var folder = YPath.simple(folderPath);
                if (!cypress.exists(folder)) {
                    continue;
                }

                var childFolders = getChildFolders(folder, cypress.get(folder), new HashSet<>());
                for (var child : childFolders) {
                    removeOldTables(cypress, thresholdTime, maxCount, child);
                }
            }
        }
    }

    private HashSet<YPath> getChildFolders(
            YPath path,
            YTreeNode node,
            HashSet<YPath> folders
    ) {
        for (var child : node.mapNode()) {
            if (child.getValue().isMapNode()) {
                getChildFolders(path.child(child.getKey()), child.getValue(), folders);
            } else {
                folders.add(path);
            }
        }

        return folders;
    }

    private void removeOldTables(Cypress cypress, long thresholdTime, long maxCount, YPath folder) {
        var tablesAttributes = cypress.get(folder, Cf.set(MODIFICATION_TIME_ATTRIBUTE, TYPE_ATTRIBUTE)).asMap();
        var tables = tablesAttributes.entrySet().stream()
                .filter(e -> isTableNode(e.getValue()))
                .map(e -> new Table(e.getKey(), parseModificationTimestamp(e.getValue())))
                .sorted(Comparator.comparingLong(Table::getModificationTime).reversed())
                .collect(toList());

        var count = 0;
        for (var table : tables) {
            count++;

            if (count > maxCount || table.getModificationTime() < thresholdTime) {
                cypress.remove(folder.child(table.getPath()));
                logger.info("Deleted table " + folder.child(table.getPath()).toString());
            }
        }
    }

    private static class Table {
        private final String path;
        private final long modificationTime;

        private Table(String path, long modificationTime) {
            this.path = path;
            this.modificationTime = modificationTime;
        }

        String getPath() {
            return path;
        }

        long getModificationTime() {
            return modificationTime;
        }
    }
}
