package ru.yandex.direct.jobs.contentcategories;

import java.time.Duration;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import com.typesafe.config.Config;
import com.typesafe.config.ConfigFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.core.entity.ppcproperty.model.PpcPropertyEnum;
import ru.yandex.direct.ytwrapper.model.YtOperator;
import ru.yandex.inside.yt.kosher.cypress.CypressNodeType;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;
import ru.yandex.misc.io.ClassPathResourceInputStreamSource;

import static java.time.temporal.ChronoUnit.DAYS;
import static ru.yandex.direct.utils.DateTimeUtils.moscowDateTimeToInstant;

@Component
@ParametersAreNonnullByDefault
public class ContentCategoriesService {
    private static final Logger logger = LoggerFactory.getLogger(ContentCategoriesService.class);
    private static final String RAW_URLS_CONFIG_CLASSPATH = "export/ppcdataexport/contentcategories/raw_urls.conf";

    static final String MODIFICATION_TIME_ATTRIBUTE = "modification_time";
    static final String TYPE_ATTRIBUTE = "type";

    private final PpcPropertiesSupport ppcPropertiesSupport;

    public ContentCategoriesService(PpcPropertiesSupport ppcPropertiesSupport) {
        this.ppcPropertiesSupport = ppcPropertiesSupport;
    }

    public void writeLastStart(PpcPropertyEnum ppcProperty, Instant start) {
        ppcPropertiesSupport.set(
                ppcProperty.getName(),
                String.valueOf(start.getEpochSecond()));
    }

    public Instant getLastOrMinStart(PpcPropertyEnum ppcProperty, int maxDaysToProcess) {
        var lastStart = readLastStart(ppcProperty);
        var minLastStart = Instant.now().minus(maxDaysToProcess, DAYS);
        if (lastStart.isBefore(minLastStart)) {
            lastStart = minLastStart;
        }
        return lastStart;
    }

    public Long getUrlsLimit(PpcPropertyEnum ppcProperty, Long defaultDayLimit) {
        var urlDayLimit = ppcPropertiesSupport.get(ppcProperty.getName());
        if (urlDayLimit == null) {
            urlDayLimit = defaultDayLimit.toString();
        }

        return Long.parseLong(urlDayLimit) * getDeltaTimeFromConfig().toHours() / 24;
    }

    private Instant readLastStart(PpcPropertyEnum ppcProperty) {
        var lastStartString = ppcPropertiesSupport.get(ppcProperty.getName());
        if (lastStartString == null) {
            return Instant.MIN;
        }

        return Instant.ofEpochSecond(Long.parseLong(lastStartString));
    }

    public ArrayList<String> getTablesList(YtOperator ytOperator, List<String> folders, Instant lastStart) {
        var tables = new ArrayList<String>();
        var cypress = ytOperator.getYt().cypress();

        for (var folder : folders) {
            var folderPath = YPath.simple(folder);
            if (!cypress.exists(folderPath)) {
                logger.warn("Yt folder {} not found", folder);
                continue;
            }

            var nodes = cypress.get(folderPath, Cf.set(MODIFICATION_TIME_ATTRIBUTE, TYPE_ATTRIBUTE)).asMap();
            logger.info("Found nodes {} in folder {}", nodes, folder);

            for (var node : nodes.entrySet()) {
                var modificationDateTimeString =
                        node.getValue().getAttribute(MODIFICATION_TIME_ATTRIBUTE).get().stringValue();
                var dateTime = Instant.parse(modificationDateTimeString);
                if (dateTime.isAfter(lastStart) && isTableNode(node.getValue())) {
                    tables.add(folderPath.child(node.getKey()).toString());
                }
            }
        }

        logger.info("Collected tables for processing {}", tables);
        return tables;
    }

    public static boolean isTableNode(YTreeNode yTreeNode) {
        return CypressNodeType.TABLE.value().equals(yTreeNode.getAttribute(TYPE_ATTRIBUTE).get().stringValue());
    }

    public static long parseModificationTimestamp(YTreeNode yTreeNode) {
        var modificationTimeString = yTreeNode.getAttribute(MODIFICATION_TIME_ATTRIBUTE).get().stringValue();
        return moscowDateTimeToInstant(
                LocalDateTime.parse(modificationTimeString, DateTimeFormatter.ISO_OFFSET_DATE_TIME))
                .getEpochSecond();
    }

    private static Duration getDeltaTimeFromConfig() {
        String configStr =
                new ClassPathResourceInputStreamSource(ContentCategoriesService.RAW_URLS_CONFIG_CLASSPATH).readText();
        Config config = ConfigFactory.parseString(configStr);
        return config.getDuration("deltaTime");
    }
}
