package ru.yandex.direct.jobs.contentcategories;

import java.time.Instant;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod;
import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.env.NonDevelopmentEnvironment;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification;
import ru.yandex.direct.juggler.check.model.CheckTag;
import ru.yandex.direct.juggler.check.model.NotificationRecipient;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectJob;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.ytwrapper.YtPathUtil;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YqlQuery;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.misc.io.ClassPathResourceInputStreamSource;

import static ru.yandex.direct.core.entity.ppcproperty.model.PpcPropertyEnum.JOBS_CONTENT_CATEGORIES_TOLOKA_OUTPUT_PROCESS_LAST_START;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_1;

/**
 * Джоба обработки результатов разметки Толоки перед мерджилкой
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(hours = 4),
        tags = {DIRECT_PRIORITY_1, CheckTag.DIRECT_PRODUCT_TEAM, CheckTag.YT},
        notifications = {
                @OnChangeNotification(recipient = NotificationRecipient.LOGIN_AMMSAID,
                        status = {JugglerStatus.OK, JugglerStatus.CRIT},
                        method = NotificationMethod.TELEGRAM),
        },
        needCheck = ProductionOnly.class
)
@Hourglass(cronExpression = "0 55 * * * ?", needSchedule = NonDevelopmentEnvironment.class)
@ParametersAreNonnullByDefault
public class ContentCategoriesTolokaOutputProcessJob extends DirectJob {

    private static final Logger logger = LoggerFactory.getLogger(ContentCategoriesTolokaOutputProcessJob.class);

    private static final YtCluster YT_CLUSTER = YtCluster.HAHN;
    private static final int MAX_DAYS_TO_PROCESS = 10;
    private static final String PROCESS_TOLOKA_OUTPUT_QUERY = String.join("\n",
            new ClassPathResourceInputStreamSource("contentcategories/process_toloka_output_table.sql").readLines());

    private final YtProvider ytProvider;
    private final ContentCategoriesService contentCategoriesService;
    private final DirectConfig tolokaConfig;

    @Autowired
    public ContentCategoriesTolokaOutputProcessJob(
            YtProvider ytProvider,
            ContentCategoriesService contentCategoriesService,
            DirectConfig directConfig
    ) {
        this.ytProvider = ytProvider;
        this.contentCategoriesService = contentCategoriesService;
        this.tolokaConfig = directConfig.getBranch("content_categories").getBranch("toloka");
    }

    @Override
    public void execute() {
        var ytOperator = ytProvider.getOperator(YT_CLUSTER);

        var lastStart = contentCategoriesService.getLastOrMinStart(
                JOBS_CONTENT_CATEGORIES_TOLOKA_OUTPUT_PROCESS_LAST_START, MAX_DAYS_TO_PROCESS);

        var start = Instant.now();

        var folders = tolokaConfig.getStringList("output_folders");
        var tables = contentCategoriesService.getTablesList(ytOperator, folders, lastStart);
        if (tables.isEmpty()) {
            logger.info("No new data found from toloka");
            return;
        }

        String processedOutputFolder = tolokaConfig.getString("processed_output_folder");
        var processedOutputTable = YtPathUtil.generatePath(processedOutputFolder,
                "processed_output_bs_" + start.getEpochSecond());

        try (var ignore = Trace.current().profile("content_categories:yql", "toloka_output_processing")) {
            ytOperator.yqlExecute(
                    new YqlQuery(
                            PROCESS_TOLOKA_OUTPUT_QUERY, String.join(",", tables), processedOutputTable)
                            .withTitle("content_categories:toloka_output_processing"));
        }

        contentCategoriesService.writeLastStart(JOBS_CONTENT_CATEGORIES_TOLOKA_OUTPUT_PROCESS_LAST_START, start);
    }
}
