package ru.yandex.direct.jobs.contentcategories;

import java.time.Instant;
import java.util.List;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableMap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod;
import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.core.entity.ppcproperty.model.PpcPropertyEnum;
import ru.yandex.direct.env.NonDevelopmentEnvironment;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification;
import ru.yandex.direct.juggler.check.model.CheckTag;
import ru.yandex.direct.juggler.check.model.NotificationRecipient;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectJob;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.ytwrapper.YtPathUtil;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YqlQuery;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.misc.io.ClassPathResourceInputStreamSource;

import static ru.yandex.direct.core.entity.ppcproperty.model.PpcPropertyEnum.JOBS_CONTENT_CATEGORIES_TOLOKA_INPUT_SECOND_LVL_LAST_START;
import static ru.yandex.direct.jobs.util.yt.YtEnvPath.relativePart;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_1_NOT_READY;

/**
 * Джоба подготовки урлов для разметки Толокой второго уровня категорий
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(hours = 4),
        tags = {DIRECT_PRIORITY_1_NOT_READY, CheckTag.DIRECT_PRODUCT_TEAM, CheckTag.YT},
        notifications = {
                @OnChangeNotification(recipient = NotificationRecipient.LOGIN_IVATKOEGOR,
                        status = {JugglerStatus.OK, JugglerStatus.CRIT},
                        method = NotificationMethod.TELEGRAM),
        },
        needCheck = ProductionOnly.class
)
@Hourglass(cronExpression = "0 50 * * * ?", needSchedule = NonDevelopmentEnvironment.class)
@ParametersAreNonnullByDefault
public class ContentCategoriesTolokaSecondLvlInputPrepareJob extends DirectJob {
    private static final Logger logger = LoggerFactory.getLogger(ContentCategoriesTolokaSecondLvlInputPrepareJob.class);

    private static final int MAX_DAYS_TO_PROCESS = 3;
    private static final YtCluster YT_CLUSTER = YtCluster.HAHN;
    private static final String YT_QUERY =
            String.join("\n", new ClassPathResourceInputStreamSource("contentcategories" +
                    "/prepare_toloka_input_table_second_lvl.sql").readLines());
    private static final Map<String, String> KEYWORDS_BY_CATEGORIES =
            ImmutableMap.<String, String>builder()
                    .put("4294968298", "travel")
                    .put("4294968302", "home")
                    .put("4294968305", "pets")
                    .put("4294968308", "food")
                    .put("4294968314", "games")
                    .put("4294968319", "books")
                    .put("4294968324", "beauty")
                    .put("4294968329", "science")
                    .put("4294968335", "sport")
                    .put("4294968342", "hobby")
                    .put("4294968345", "shopping")
                    .put("4294968349", "family")
                    .build();

    private final YtProvider ytProvider;
    private final ContentCategoriesService contentCategoriesService;
    private final DirectConfig tolokaConfig;

    public ContentCategoriesTolokaSecondLvlInputPrepareJob(
            YtProvider ytProvider,
            ContentCategoriesService contentCategoriesService,
            DirectConfig directConfig
    ) {
        this.ytProvider = ytProvider;
        this.contentCategoriesService = contentCategoriesService;
        this.tolokaConfig = directConfig.getBranch("content_categories").getBranch("toloka");
    }

    @Override
    public void execute() {
        var ytOperator = ytProvider.getOperator(YT_CLUSTER);
        var ytClusterConfig = ytProvider.getClusterConfig(YT_CLUSTER);

        var lastStart = contentCategoriesService.getLastOrMinStart(
                JOBS_CONTENT_CATEGORIES_TOLOKA_INPUT_SECOND_LVL_LAST_START, MAX_DAYS_TO_PROCESS);

        var basePath = YtPathUtil.generatePath(ytProvider.getClusterConfig(YT_CLUSTER).getHome(), relativePart());
        var start = Instant.now();

        var sourceFolder = tolokaConfig.getString("raw_urls_folder");
        var tables = contentCategoriesService.getTablesList(ytOperator, List.of(sourceFolder), lastStart);
        if (tables.isEmpty()) {
            logger.info("No new data found for toloka second lvl");
            return;
        }

        var defaultDayLimit = tolokaConfig.getLong("url_day_limit");
        var rowsLimit = contentCategoriesService.getUrlsLimit(
                PpcPropertyEnum.JOBS_CONTENT_CATEGORIES_TOLOKA_SECOND_LVL_URLS_DAY_LIMIT, defaultDayLimit);

        try (var ignore = Trace.current().profile("content_categories:yql", "toloka_input_second_lvl")) {
            for (var entry : KEYWORDS_BY_CATEGORIES.entrySet()) {
                var categoryId = entry.getKey();
                var folderName = entry.getValue();

                ytOperator.yqlExecute(
                        new YqlQuery(
                                YT_QUERY,
                                ytClusterConfig.getUser(),
                                tables.get(0),
                                categoryId,
                                folderName,
                                basePath,
                                rowsLimit).withTitle("content_categories:toloka_input_second_lvl"));
            }
        }
        contentCategoriesService.writeLastStart(JOBS_CONTENT_CATEGORIES_TOLOKA_INPUT_SECOND_LVL_LAST_START, start);
    }
}
