package ru.yandex.direct.jobs.contentcategories;

import java.time.Instant;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod;
import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.env.TypicalEnvironment;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification;
import ru.yandex.direct.juggler.check.model.CheckTag;
import ru.yandex.direct.juggler.check.model.NotificationRecipient;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectJob;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.ytwrapper.YtPathUtil;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YqlQuery;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.misc.io.ClassPathResourceInputStreamSource;

import static ru.yandex.direct.core.entity.ppcproperty.model.PpcPropertyEnum.JOBS_CONTENT_CATEGORIES_YACOFAST_DATA_COLLECTOR_LAST_START;
import static ru.yandex.direct.jobs.util.yt.YtEnvPath.relativePart;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_1;

/**
 * Джоба выгрузки урлов, размеченных категориями контента, в yacofast
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(days = 1),
        tags = {DIRECT_PRIORITY_1, CheckTag.DIRECT_PRODUCT_TEAM, CheckTag.YT},
        notifications = {
                @OnChangeNotification(recipient = NotificationRecipient.LOGIN_IVATKOEGOR,
                        status = {JugglerStatus.OK, JugglerStatus.CRIT},
                        method = NotificationMethod.TELEGRAM),
        },
        needCheck = ProductionOnly.class
)
@Hourglass(cronExpression = "0 0 * ? * *", needSchedule = TypicalEnvironment.class)
@ParametersAreNonnullByDefault
class ContentCategoriesYacofastDataCollectorJob extends DirectJob {
    private static final Logger logger = LoggerFactory.getLogger(ContentCategoriesYacofastDataCollectorJob.class);

    private static final YtCluster YT_CLUSTER = YtCluster.HAHN;
    private static final int MAX_DAYS_TO_PROCESS = 3;
    private static final String YACOFAST_DATA_QUERY = String.join("\n",
            new ClassPathResourceInputStreamSource("contentcategories/yacofast_data.sql").readLines());

    private final YtProvider ytProvider;
    private final ContentCategoriesService contentCategoriesService;
    private final DirectConfig config;

    public ContentCategoriesYacofastDataCollectorJob(
            YtProvider ytProvider,
            ContentCategoriesService contentCategoriesService,
            DirectConfig config
    ) {
        this.ytProvider = ytProvider;
        this.contentCategoriesService = contentCategoriesService;
        this.config = config;
    }

    @Override
    public void execute() {
        var ytOperator = ytProvider.getOperator(YT_CLUSTER);

        var lastStart = contentCategoriesService.getLastOrMinStart(
                JOBS_CONTENT_CATEGORIES_YACOFAST_DATA_COLLECTOR_LAST_START, MAX_DAYS_TO_PROCESS);

        var basePath = YtPathUtil.generatePath(ytProvider.getClusterConfig(YT_CLUSTER).getHome(), relativePart());
        var start = Instant.now();

        var folders = config.getStringList("content_categories.yacofast_data_collector.source_folders");
        var tables = contentCategoriesService.getTablesList(ytOperator, folders, lastStart);
        if (tables.isEmpty()) {
            logger.info("No new data found for yacofast");
            return;
        }

        try (var ignore = Trace.current().profile("content_categories:yql", "yacofast_data_collector")) {
            ytOperator.yqlExecute(
                    new YqlQuery(
                            YACOFAST_DATA_QUERY,
                            String.join(",", tables),
                            basePath).withTitle("content_categories:yacofast_data_collector"));
        }

        contentCategoriesService.writeLastStart(JOBS_CONTENT_CATEGORIES_YACOFAST_DATA_COLLECTOR_LAST_START, start);
    }
}
