package ru.yandex.direct.jobs.contentcategories;

import java.time.Instant;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableMap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod;
import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.core.entity.ppcproperty.model.PpcPropertyEnum;
import ru.yandex.direct.env.NonDevelopmentEnvironment;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification;
import ru.yandex.direct.juggler.check.model.CheckTag;
import ru.yandex.direct.juggler.check.model.NotificationRecipient;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectJob;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.ytwrapper.YtPathUtil;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YqlQuery;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.direct.ytwrapper.model.YtSQLSyntaxVersion;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.misc.io.ClassPathResourceInputStreamSource;

import static ru.yandex.direct.core.entity.ppcproperty.model.PpcPropertyEnum.JOBS_CONTENT_CATEGORIES_YANG_INPUT_PREPARE_LAST_START;

/**
 * Джоба подготовки урлов для разметки асессорами в Янге.
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(hours = 12),
        tags = {/*DIRECT_PRIORITY_1,*/ CheckTag.DIRECT_PRODUCT_TEAM, CheckTag.YT},
        notifications = {
                @OnChangeNotification(recipient = NotificationRecipient.LOGIN_AMMSAID,
                        status = {JugglerStatus.OK, JugglerStatus.CRIT},
                        method = NotificationMethod.TELEGRAM),
        },
        needCheck = ProductionOnly.class
)

@Hourglass(periodInSeconds = 3 * 60 * 60, needSchedule = NonDevelopmentEnvironment.class)
@ParametersAreNonnullByDefault
public class ContentCategoriesYangInputPrepareJob extends DirectJob {

    private static final Logger logger = LoggerFactory.getLogger(ContentCategoriesYangInputPrepareJob.class);
    private static final YtCluster YT_CLUSTER = YtCluster.HAHN;
    private static final int MAX_DAYS_TO_PROCESS = 3;
    private static final String PREPARE_SERVICES_MONITORING_INPUT_QUERY = String.join("\n",
            new ClassPathResourceInputStreamSource("contentcategories/prepare_services_monitoring_input_table.sql").readLines());
    private static final String PREPARE_TOTAL_MONITORING_INPUT_QUERY = String.join("\n",
            new ClassPathResourceInputStreamSource("contentcategories/prepare_total_monitoring_input_table.sql").readLines());
    private static final Map<String, String> CATEGORIES_GROUP_FOLDERS = ImmutableMap.<String, String>builder()
            .put("tobacco_video-games_gambling_alcohol", "yang")
            .put("negative_profanity_childrens_adult", "yang")
            .put("weapons_sharing_terrorism_contravenes-legislation", "yang")
            .put("religion_occultism_medical_dating", "yang")
            .put("tragedy_politics_news", "yang")
            .put("auto_business_travel_home", "yang")
            .put("pets_food_law_games", "yang")
            .put("internet_arts_books", "yang")
            .put("electronics_beauty_society_science", "yang")
            .put("estate_news_job_education", "yang")
            .put("sport_family_finance_hobby_shopping", "yang")
            .put("travel", "yang_second_lvl")
            .put("home", "yang_second_lvl")
            .put("pets", "yang_second_lvl")
            .put("food", "yang_second_lvl")
            .put("games", "yang_second_lvl")
            .put("books", "yang_second_lvl")
            .put("beauty", "yang_second_lvl")
            .put("science", "yang_second_lvl")
            .put("sport", "yang_second_lvl")
            .put("hobby", "yang_second_lvl")
            .put("shopping", "yang_second_lvl")
            .put("family", "yang_second_lvl")
            .build();

    private final YtProvider ytProvider;
    private final ContentCategoriesService contentCategoriesService;
    private final DirectConfig yangConfig;

    @Autowired
    public ContentCategoriesYangInputPrepareJob(
            YtProvider ytProvider,
            ContentCategoriesService contentCategoriesService,
            DirectConfig directConfig
    ) {
        this.ytProvider = ytProvider;
        this.contentCategoriesService = contentCategoriesService;
        this.yangConfig = directConfig.getBranch("content_categories").getBranch("yang");
    }

    @Override
    public void execute() {
        var ytOperator = ytProvider.getOperator(YT_CLUSTER, YtSQLSyntaxVersion.SQLv1);
        var ytClusterConfig = ytProvider.getClusterConfig(YT_CLUSTER);
        var folderByService = yangConfig.getBranch("services_folders").asMap();

        Long defaultDayLimit = yangConfig.getLong("url_day_limit");
        Long tableLimit = contentCategoriesService.getUrlsLimit(
                PpcPropertyEnum.JOBS_CONTENT_CATEGORIES_YANG_URLS_DAY_LIMIT, defaultDayLimit) / folderByService.keySet().size();

        var lastStart = contentCategoriesService.getLastOrMinStart(
                JOBS_CONTENT_CATEGORIES_YANG_INPUT_PREPARE_LAST_START, MAX_DAYS_TO_PROCESS);

        var start = Instant.now();

        var yangInputPath = yangConfig.getString("input_path");
        var pathParts = new ArrayList<>(CATEGORIES_GROUP_FOLDERS.entrySet()).get(0);
        var yangInputTable = YtPathUtil.generatePath(yangInputPath,
                pathParts.getValue(),
                pathParts.getKey(),
                "input_yang_" + start.getEpochSecond());

        for (var entry : folderByService.entrySet()) {
            String service = entry.getKey();
            String folder = entry.getValue();

            String query = "Total".equals(service) ? PREPARE_TOTAL_MONITORING_INPUT_QUERY :
                    PREPARE_SERVICES_MONITORING_INPUT_QUERY;
            String queryTag = service + "_monitoring_input_prepare";

            var tables = contentCategoriesService.getTablesList(ytOperator, Collections.singletonList(folder),
                    lastStart);
            if (tables.isEmpty()) {
                logger.info("No new data found for " + service);
                continue;
            }

            try (var ignore = Trace.current().profile("content_categories:yql", queryTag)) {
                ytOperator.yqlExecute(
                        new YqlQuery(
                                query, ytClusterConfig.getUser(),
                                String.join(",", tables), yangInputTable, service, tableLimit)
                                .withTitle("content_categories:" + queryTag));
            }
        }

        var cypress = ytOperator.getYt().cypress();
        if (cypress.exists(YPath.simple(yangInputTable))) {
            for (var groupPath : CATEGORIES_GROUP_FOLDERS.entrySet()) {
                var yangInputCopiedTable = YtPathUtil.generatePath(yangInputPath,
                        groupPath.getValue(),
                        groupPath.getKey(),
                        "input_yang_" + start.getEpochSecond());
                if (yangInputCopiedTable.equals(yangInputTable)) {
                    continue;
                }

                try {
                    cypress.copy(
                            YPath.simple(yangInputTable),
                            YPath.simple(yangInputCopiedTable),
                            true,
                            false,
                            false);
                    logger.info("{} was copied to {}", yangInputTable, yangInputCopiedTable);
                } catch (Exception e) {
                    logger.error("Can't copy table from " + yangInputTable + " to " + yangInputCopiedTable, e);
                }
            }
        }

        contentCategoriesService.writeLastStart(JOBS_CONTENT_CATEGORIES_YANG_INPUT_PREPARE_LAST_START, start);
    }
}
