package ru.yandex.direct.jobs.contentpromotion.collection.receivechanges;

import java.math.BigInteger;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.banner.repository.BannerCommonRepository;
import ru.yandex.direct.core.entity.banner.type.contentpromo.BannerWithContentPromotionRepository;
import ru.yandex.direct.core.entity.contentpromotion.model.ContentPromotionContent;
import ru.yandex.direct.core.entity.contentpromotion.repository.ContentPromotionRepository;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.env.EnvironmentType;
import ru.yandex.direct.jobs.contentpromotion.collection.receivechanges.model.CollectionChangeInfo;
import ru.yandex.direct.jobs.contentpromotion.collection.receivechanges.model.CollectionChangeStatus;
import ru.yandex.direct.libs.collections.CollectionsClient;
import ru.yandex.direct.libs.collections.model.serpdata.CollectionSerpDataResult;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.utils.HashingUtils;

import static ru.yandex.direct.jobs.contentpromotion.ContentPromotionUtils.convertToContentPromotionContent;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Сервис для обработки изменений коллекций.
 */
@Service
@ParametersAreNonnullByDefault
public class CollectionsChangesService {

    private static final Logger logger = LoggerFactory.getLogger(CollectionsChangesService.class);

    private final EnvironmentType environmentType;
    private final Collection<Integer> dbShards;
    private final BannerWithContentPromotionRepository bannerContentPromotionRepository;
    private final ContentPromotionRepository contentPromotionRepository;
    private final CollectionsClient collectionsClient;
    private final DslContextProvider dslContextProvider;
    private final BannerCommonRepository bannerCommonRepository;

    @Autowired
    public CollectionsChangesService(EnvironmentType environmentType, ShardHelper shardHelper,
                                     BannerWithContentPromotionRepository bannerContentPromotionRepository,
                                     ContentPromotionRepository contentPromotionRepository,
                                     CollectionsClient collectionsClient, DslContextProvider dslContextProvider,
                                     BannerCommonRepository bannerCommonRepository) {
        dbShards = shardHelper.dbShards();

        this.environmentType = environmentType;
        this.bannerContentPromotionRepository = bannerContentPromotionRepository;
        this.contentPromotionRepository = contentPromotionRepository;
        this.collectionsClient = collectionsClient;
        this.dslContextProvider = dslContextProvider;
        this.bannerCommonRepository = bannerCommonRepository;
    }

    /**
     * Обрабатывает изменение доступности коллекции.
     *
     * @param event информация об изменении коллекции
     */
    void processCollectionAccessibilityChange(CollectionChangeInfo event) {
        String boardId = event.getBoardId();
        boolean isInaccessible = event.getCollectionChangeStatus() != CollectionChangeStatus.COLLECTION_OPENED;

        dbShards.forEach(
                shard -> contentPromotionRepository.updateCollectionAccessibilityStatus(shard, boardId, isInaccessible));
    }

    /**
     * Обрабатывает изменение коллекции (добавление/удаление карточек и т.п.).
     *
     * @param event информация об изменении коллекции
     */
    void processCollectionUpdate(CollectionChangeInfo event) {
        String boardId = event.getBoardId();

        CollectionSerpDataResult collectionSerpDataResult = collectionsClient.getCollectionSerpDataResult(boardId);

        if (!collectionSerpDataResult.isSuccessfulResult()) {
            logger.error("Got error during collection serp data request: {}",
                    collectionSerpDataResult.getError().getMessage());
            return;
        }

        ContentPromotionContent updatedCollection =
                convertToContentPromotionContent(collectionSerpDataResult.getCollectionSerpData(), environmentType);

        dbShards.forEach(shard -> processCollectionUpdateInShard(shard, boardId, updatedCollection));
    }

    private void processCollectionUpdateInShard(int shard, String boardId, ContentPromotionContent updatedCollection) {
        List<ContentPromotionContent> collectionsToUpdate =
                contentPromotionRepository.getCollections(shard, boardId);

        if (collectionsToUpdate.isEmpty()) {
            return;
        }

        List<AppliedChanges<ContentPromotionContent>> appliedChanges = StreamEx.of(collectionsToUpdate)
                .map(collectionToUpdate -> convertToAppliedChanges(collectionToUpdate, updatedCollection))
                .toList();

        Map<Long, List<Long>> nonDraftBannerIdsByContentId =
                bannerContentPromotionRepository.getNonDraftBannerIdsByContentIds(shard,
                        mapList(collectionsToUpdate, ContentPromotionContent::getId));
        Set<Long> bannerIdsToBsSync = StreamEx.of(nonDraftBannerIdsByContentId.values())
                .flatMap(StreamEx::of)
                .toSet();

        dslContextProvider.ppc(shard).transaction(configuration -> {
            contentPromotionRepository.updateContentPromotionContents(configuration.dsl(), appliedChanges);
            bannerCommonRepository.resetStatusBsSyncedByIds(configuration.dsl(), bannerIdsToBsSync);
        });
    }

    private AppliedChanges<ContentPromotionContent> convertToAppliedChanges(ContentPromotionContent before,
                                                                            ContentPromotionContent after) {
        ModelChanges<ContentPromotionContent> changes = new ModelChanges<>(before.getId(),
                ContentPromotionContent.class);

        // cleanup 4-byte characters
        String metadataAfter = after.getMetadata().replaceAll("[^\\u0000-\\uFFFF]", "\uFFFD");

        BigInteger metadataHashAfter = HashingUtils.getMd5HalfHashUtf8(metadataAfter);

        if (!Objects.equals(before.getMetadataHash(), metadataHashAfter)) {
            changes.process(metadataHashAfter, ContentPromotionContent.METADATA_HASH);
            changes.process(metadataAfter, ContentPromotionContent.METADATA);
            changes.process(after.getPreviewUrl(), ContentPromotionContent.PREVIEW_URL);
        } else {
            logger.error("Collection with external id {} is not updated", before.getExternalId());
        }

        return changes.applyTo(before);
    }
}
