package ru.yandex.direct.jobs.contentpromotion.collection.receivechanges;

import java.util.List;
import java.util.function.Supplier;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.binlogbroker.logbroker_utils.reader.LogbrokerBatchReader;
import ru.yandex.direct.binlogbroker.logbroker_utils.reader.LogbrokerReaderCloseException;
import ru.yandex.direct.binlogbroker.logbroker_utils.reader.RetryingLogbrokerBatchReader;
import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.env.EnvironmentType;
import ru.yandex.direct.ess.common.logbroker.LogbrokerClientFactoryFacade;
import ru.yandex.direct.ess.common.logbroker.LogbrokerConsumerProperties;
import ru.yandex.direct.jobs.contentpromotion.collection.UpdateCollectionsPropertyHolder;
import ru.yandex.direct.jobs.contentpromotion.collection.receivechanges.model.CollectionChangeInfo;
import ru.yandex.direct.jobs.contentpromotion.collection.receivechanges.model.CollectionChangeStatus;
import ru.yandex.direct.scheduler.support.DirectJob;
import ru.yandex.direct.tvm.TvmIntegration;
import ru.yandex.direct.utils.InterruptedRuntimeException;
import ru.yandex.kikimr.persqueue.auth.Credentials;
import ru.yandex.kikimr.persqueue.consumer.SyncConsumer;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.registry.MetricRegistry;

import static ru.yandex.direct.jobs.receiveorganizationstatuschanges.logbroker.ReceiveChangesLogbrokerUtils.createConsumerProperties;
import static ru.yandex.direct.jobs.receiveorganizationstatuschanges.logbroker.ReceiveChangesLogbrokerUtils.createCredentialsSupplier;
import static ru.yandex.direct.jobs.receiveorganizationstatuschanges.logbroker.ReceiveChangesLogbrokerUtils.createSolomonRegistryLabels;
import static ru.yandex.direct.solomon.SolomonUtils.SOLOMON_REGISTRY;

/**
 * Джоба для получения и обработки изменений коллекций.
 */
//@JugglerCheck(ttl = @JugglerCheck.Duration(hours = 1),
//        needCheck = ProductionOnly.class,
//        tags = {DIRECT_API_TEAM},
//        notifications = @OnChangeNotification(
//                recipient = NotificationRecipient.CHAT_API_MONITORING,
//                method = NotificationMethod.TELEGRAM,
//                status = {JugglerStatus.OK, JugglerStatus.CRIT}
//        )
//)
//@JugglerCheck(ttl = @JugglerCheck.Duration(hours = 1),
//        needCheck = NonProductionEnvironment.class,
//        tags = {DIRECT_API_TEAM, JOBS_RELEASE_REGRESSION}
//)
// Выключена в связи с закрытием Коллекций
//@Hourglass(periodInSeconds = 5 * 60, needSchedule = TypicalEnvironment.class)
@ParametersAreNonnullByDefault
public class ReceiveCollectionsChangesJob extends DirectJob {

    private static final Logger logger = LoggerFactory.getLogger(ReceiveCollectionsChangesJob.class);

    private final Runnable initializer;

    private LogbrokerBatchReader<CollectionChangeInfo> logbrokerReader;
    private Labels solomonRegistryLabels;

    private final CollectionsChangesService collectionsChangesService;
    private final UpdateCollectionsPropertyHolder updateCollectionsJobsPropertyHolder;

    @Autowired
    public ReceiveCollectionsChangesJob(DirectConfig directConfig, TvmIntegration tvmIntegration,
                                        CollectionsChangesService collectionsChangesService,
                                        PpcPropertiesSupport ppcPropertiesSupport, EnvironmentType environmentType) {
        this.initializer = createInitializer(directConfig, tvmIntegration);

        this.collectionsChangesService = collectionsChangesService;
        this.updateCollectionsJobsPropertyHolder =
                new UpdateCollectionsPropertyHolder(ppcPropertiesSupport, environmentType);
    }

    private Runnable createInitializer(DirectConfig directConfig, TvmIntegration tvmIntegration) {
        return () -> {
            DirectConfig collectionsConfig = directConfig.getBranch("collections");

            Supplier<Credentials> credentialsSupplier = createCredentialsSupplier(collectionsConfig, tvmIntegration);
            LogbrokerClientFactoryFacade logbrokerClientFactory = new LogbrokerClientFactoryFacade(credentialsSupplier);

            LogbrokerConsumerProperties consumerProperties = createConsumerProperties(collectionsConfig);
            Supplier<SyncConsumer> syncConsumerSupplier =
                    logbrokerClientFactory.createConsumerSupplier(consumerProperties);

            solomonRegistryLabels = createSolomonRegistryLabels("collections_changes_reader",
                    consumerProperties.getReadTopic());
            MetricRegistry sensorsRegistry = SOLOMON_REGISTRY.subRegistry(solomonRegistryLabels);

            logbrokerReader = new RetryingLogbrokerBatchReader<>(
                    () -> new CollectionsChangesLogbrokerReader(syncConsumerSupplier, false, sensorsRegistry),
                    consumerProperties.getRetries());
        };
    }

    private void init() {
        logger.info("Initializing ReceiveCollectionsChangesJob");
        initializer.run();
    }

    @Override
    public void execute() {
        if (!updateCollectionsJobsPropertyHolder.isReceiveCollectionsChangesJobEnabled()) {
            return;
        }

        init();
        try {
            logbrokerReader.fetchEvents(this::processEvents);
        } catch (InterruptedException ex) {
            Thread.currentThread().interrupt();
            throw new InterruptedRuntimeException(ex);
        }
    }

    void processEvents(List<CollectionChangeInfo> events) {
        logger.info("Fetched {} events: {}", events.size(), events);

        events.forEach(this::processEvent);
    }

    private void processEvent(CollectionChangeInfo event) {
        CollectionChangeStatus collectionChangeStatus = event.getCollectionChangeStatus();

        switch (collectionChangeStatus) {
            case COLLECTION_OPENED:
            case COLLECTION_CLOSED:
            case COLLECTION_DELETED:
                collectionsChangesService.processCollectionAccessibilityChange(event);
                break;
            case COLLECTION_UPDATED:
                collectionsChangesService.processCollectionUpdate(event);
                break;
            default:
                throw new IllegalArgumentException("Unsupported event type: " + collectionChangeStatus.name());
        }
    }

    @Override
    public void finish() {
        if (solomonRegistryLabels != null) {
            SOLOMON_REGISTRY.removeSubRegistry(solomonRegistryLabels);
        }
        if (logbrokerReader != null) {
            try {
                logbrokerReader.close();
            } catch (LogbrokerReaderCloseException ex) {
                logger.error("Error while closing logbrokerReader", ex);
            }
        }
    }
}
