package ru.yandex.direct.jobs.contentpromotion.collection.updatecontent;

import java.util.List;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Preconditions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.core.entity.banner.repository.BannerCommonRepository;
import ru.yandex.direct.core.entity.banner.type.contentpromo.BannerWithContentPromotionRepository;
import ru.yandex.direct.core.entity.contentpromotion.model.ContentPromotionContent;
import ru.yandex.direct.core.entity.contentpromotion.model.ContentPromotionContentType;
import ru.yandex.direct.core.entity.contentpromotion.repository.ContentPromotionRepository;
import ru.yandex.direct.core.entity.contentpromotion.type.ContentPromotionCoreTypeSupportFacade;
import ru.yandex.direct.core.entity.contentpromotion.type.collection.ContentPromotionCollectionUtils;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.env.EnvironmentType;
import ru.yandex.direct.jobs.contentpromotion.collection.UpdateCollectionsPropertyHolder;
import ru.yandex.direct.jobs.contentpromotion.common.AbstractUpdateContentJob;
import ru.yandex.direct.jobs.contentpromotion.common.ContentPromotionJobConfiguration;
import ru.yandex.direct.libs.collections.CollectionsClient;
import ru.yandex.direct.libs.collections.model.serpdata.CollectionSerpData;
import ru.yandex.direct.libs.collections.model.serpdata.CollectionSerpDataResult;

import static java.util.Collections.singletonList;

/**
 * Обновляет данные продвижения коллекций в таблице ppc.content_promotion
 */
// Выключена в связи с закрытием Коллекций
//@Hourglass(periodInSeconds = 60 * 20)
@ParametersAreNonnullByDefault
public class UpdateContentCollectionsJob extends AbstractUpdateContentJob {
    private static final Logger LOGGER = LoggerFactory.getLogger(UpdateContentCollectionsJob.class);

    private final CollectionsClient collectionsClient;
    private final EnvironmentType environmentType;
    private final UpdateCollectionsPropertyHolder updateCollectionsJobsPropertyHolder;

    @Autowired
    public UpdateContentCollectionsJob(BannerWithContentPromotionRepository bannerContentPromotionRepository,
                                       ContentPromotionRepository contentPromotionRepository,
                                       BannerCommonRepository bannerCommonRepository,
                                       ContentPromotionCoreTypeSupportFacade contentPromotionCoreTypeSupportFacade,
                                       DslContextProvider dslContextProvider,
                                       PpcPropertiesSupport ppcPropertiesSupport,
                                       CollectionsClient collectionsClient,
                                       EnvironmentType environmentType) {
        super(bannerContentPromotionRepository, contentPromotionRepository, bannerCommonRepository,
                contentPromotionCoreTypeSupportFacade, dslContextProvider);

        this.collectionsClient = collectionsClient;
        this.environmentType = environmentType;
        this.updateCollectionsJobsPropertyHolder =
                new UpdateCollectionsPropertyHolder(ppcPropertiesSupport, environmentType);
    }

    public UpdateContentCollectionsJob(int shard,
                                       BannerWithContentPromotionRepository bannerContentPromotionRepository,
                                       ContentPromotionRepository contentPromotionRepository,
                                       BannerCommonRepository bannerCommonRepository,
                                       ContentPromotionCoreTypeSupportFacade contentPromotionCoreTypeSupportFacade,
                                       DslContextProvider dslContextProvider,
                                       PpcPropertiesSupport ppcPropertiesSupport,
                                       CollectionsClient collectionsClient,
                                       EnvironmentType environmentType) {
        super(shard, bannerContentPromotionRepository, contentPromotionRepository, bannerCommonRepository,
                contentPromotionCoreTypeSupportFacade, dslContextProvider);

        this.collectionsClient = collectionsClient;
        this.environmentType = environmentType;
        this.updateCollectionsJobsPropertyHolder =
                new UpdateCollectionsPropertyHolder(ppcPropertiesSupport, environmentType);
    }

    @Override
    protected ContentPromotionJobConfiguration getJobConfiguration() {
        return new ContentPromotionCollectionJobConfiguration(
                () -> updateCollectionsJobsPropertyHolder.isUpdateContentCollectionsJobEnabled());
    }

    @Override
    protected void log(String messageFormat, Object... arguments) {
        LOGGER.info(messageFormat, arguments);
    }


    @Override
    protected ContentPromotionContentType getContentType() {
        return ContentPromotionContentType.COLLECTION;
    }

    @Override
    protected List<ContentPromotionContent> getNewValues(@Nonnull List<ContentPromotionContent> oldValues) {
        Preconditions.checkState(oldValues.size() == 1, "Don't update more than one collection per request");
        return singletonList(getSingleNewValue(oldValues.get(0)));
    }

    private ContentPromotionContent getSingleNewValue(ContentPromotionContent oldValue) {
        CollectionSerpDataResult collectionSerpDataResult =
                collectionsClient.getCollectionSerpDataResult(oldValue.getExternalId());

        if (collectionSerpDataResult.isSuccessfulResult()) {
            return convertToContentPromotionContent(collectionSerpDataResult.getCollectionSerpData());
        }

        if (collectionSerpDataResult.isCollectionNotFoundResult()) {
            return null;
        }

        LOGGER.error("Got error during collection serp data request: {}",
                collectionSerpDataResult.getError().getMessage());
        return oldValue;
    }

    private ContentPromotionContent convertToContentPromotionContent(@Nonnull CollectionSerpData serpData) {
        return new ContentPromotionContent()
                .withMetadata(serpData.getNormalizedJson())
                .withPreviewUrl(ContentPromotionCollectionUtils.convertToPreviewUrl(serpData.getThumbId(),
                        environmentType));
    }
}

