package ru.yandex.direct.jobs.contentpromotion.common;

import java.math.BigInteger;
import java.time.LocalDateTime;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import com.google.common.base.Preconditions;
import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.banner.repository.BannerCommonRepository;
import ru.yandex.direct.core.entity.banner.type.contentpromo.BannerWithContentPromotionRepository;
import ru.yandex.direct.core.entity.contentpromotion.model.ContentPromotionContent;
import ru.yandex.direct.core.entity.contentpromotion.model.ContentPromotionContentType;
import ru.yandex.direct.core.entity.contentpromotion.repository.ContentPromotionRepository;
import ru.yandex.direct.core.entity.contentpromotion.type.ContentPromotionCoreTypeSupportFacade;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.scheduler.support.DirectShardedJob;

import static com.google.common.collect.Lists.partition;
import static java.util.function.Function.identity;

/**
 * Базовый класс для джобы по обновлению контента на баннерах типа content_promotion и content_promotion_video
 * Берёт пачку контента из шарда (количество оговорено в конфигурации джобы), ходит во внешний сервис
 * и обновляет его в базе
 */
public abstract class AbstractUpdateContentJob extends DirectShardedJob {

    private final BannerWithContentPromotionRepository bannerContentPromotionRepository;
    private final ContentPromotionRepository contentPromotionRepository;
    private final BannerCommonRepository bannerCommonRepository;
    private final ContentPromotionCoreTypeSupportFacade contentPromotionCoreTypeSupportFacade;
    private final DslContextProvider dslContextProvider;
    private final ContentPromotionJobConfiguration jobOptions;

    /**
     * Экземпляр ContentPromotionJobConfiguration с конфигурацией запуска джобы
     */
    protected abstract ContentPromotionJobConfiguration getJobConfiguration();

    /**
     * Функция для записи сообщения в лог
     * Пишем обновляемый контент и полученный из внешнего сервиса
     */
    protected abstract void log(String messageFormat, Object... arguments);

    /**
     * Тип обновляемого джобой контента
     */
    protected abstract ContentPromotionContentType getContentType();

    /**
     * Получение нового контента по старому
     * Должно внутри себя вызывать поход во внешний сервис
     *
     * @param oldValues - контент, который надо обновить
     */
    protected abstract List<ContentPromotionContent> getNewValues(@Nonnull List<ContentPromotionContent> oldValues);

    public AbstractUpdateContentJob(int shard,
                                    BannerWithContentPromotionRepository bannerContentPromotionRepository,
                                    ContentPromotionRepository contentPromotionRepository,
                                    BannerCommonRepository bannerCommonRepository,
                                    ContentPromotionCoreTypeSupportFacade contentPromotionCoreTypeSupportFacade,
                                    DslContextProvider dslContextProvider) {
        super(shard);
        this.jobOptions = getJobConfiguration();
        this.bannerContentPromotionRepository = bannerContentPromotionRepository;
        this.contentPromotionRepository = contentPromotionRepository;
        this.bannerCommonRepository = bannerCommonRepository;
        this.contentPromotionCoreTypeSupportFacade = contentPromotionCoreTypeSupportFacade;
        this.dslContextProvider = dslContextProvider;
    }

    public AbstractUpdateContentJob(BannerWithContentPromotionRepository bannerContentPromotionRepository,
                                    ContentPromotionRepository contentPromotionRepository,
                                    BannerCommonRepository bannerCommonRepository,
                                    ContentPromotionCoreTypeSupportFacade contentPromotionCoreTypeSupportFacade,
                                    DslContextProvider dslContextProvider) {
        this.jobOptions = getJobConfiguration();
        this.bannerContentPromotionRepository = bannerContentPromotionRepository;
        this.contentPromotionRepository = contentPromotionRepository;
        this.bannerCommonRepository = bannerCommonRepository;
        this.contentPromotionCoreTypeSupportFacade = contentPromotionCoreTypeSupportFacade;
        this.dslContextProvider = dslContextProvider;
    }

    @Override
    public void execute() {
        if (jobOptions.doJob()) {
            executeInternal(getShard());
        }
    }

    private void executeInternal(int shard) {
        List<ContentPromotionContent> contentToRefresh = contentPromotionRepository
                .getContentPromotionContentToUpdate(shard, getContentType(),
                        jobOptions.getRefreshCount(), jobOptions.getGracePeriod());
        log("Got {} content to check: {}", contentToRefresh.size(),
                StreamEx.of(contentToRefresh).map(ContentPromotionContent::getUrl).joining(", "));

        LocalDateTime now = LocalDateTime.now();
        List<AppliedChanges<ContentPromotionContent>> appliedChanges =
                StreamEx.of(partition(contentToRefresh, jobOptions.getChunkSize()))
                        .map(chunk -> getChunkChanges(chunk, now))
                        .toFlatList(identity());

        log("Resulting content: {}", StreamEx.of(appliedChanges)
                .map(AppliedChanges::getModel).map(ContentPromotionContent::toString).joining(", "));

        List<BannerUpdateRequest> bannerUpdateRequests = StreamEx.of(appliedChanges)
                .map(BannerUpdateRequest::fromContentPromotionChanges)
                .filter(BannerUpdateRequest::needToUpdateBanner)
                .toList();
        List<Long> updatedContentIds = StreamEx.of(bannerUpdateRequests)
                .map(BannerUpdateRequest::getContentId)
                .toList();
        Map<Long, List<Long>> nonDraftBannerIdsByContentId = bannerContentPromotionRepository
                .getNonDraftBannerIdsByContentIds(shard, updatedContentIds);

        List<Long> bannerIdsToBsSync = StreamEx.of(bannerUpdateRequests)
                .filter(BannerUpdateRequest::isShouldSyncToBs)
                .map(BannerUpdateRequest::getContentId)
                .map(nonDraftBannerIdsByContentId::get)
                .nonNull() // Контент может быть не привязан к баннеру
                .toFlatList(identity());
        dslContextProvider.ppc(shard).transaction(configuration -> {
            contentPromotionRepository.updateContentPromotionContents(configuration.dsl(), appliedChanges);
            bannerCommonRepository.resetStatusBsSyncedByIds(configuration.dsl(), bannerIdsToBsSync);
        });
    }

    private List<AppliedChanges<ContentPromotionContent>> getChunkChanges(
            @Nonnull List<ContentPromotionContent> contentPromotionContents,
            LocalDateTime updateTime) {
        List<ContentPromotionContent> contentPromotionContentsAfter = getNewValues(contentPromotionContents);
        Preconditions.checkState(contentPromotionContents.size() == contentPromotionContentsAfter.size(),
                "content promotion list sizes must match");
        return StreamEx.of(contentPromotionContents)
                .zipWith(contentPromotionContentsAfter.stream(),
                        (before, after) -> convertToAppliedChanges(before, after, updateTime))
                .toList();
    }

    //Смотрим в after на поля metadata, preview_url
    private AppliedChanges<ContentPromotionContent> convertToAppliedChanges(ContentPromotionContent before,
                                                                            @Nullable ContentPromotionContent after,
                                                                            LocalDateTime updateTime) {
        ModelChanges<ContentPromotionContent> changes = new ModelChanges<>(before.getId(),
                ContentPromotionContent.class);

        // Вернулась неожиданная ошибка на запрос. Неожиданными являются все, кроме 404.
        // TODO: эта логика работает только для коллекций, нужно сделать и для видео
        boolean requestFailed = before == after;
        boolean isAccessibleBefore = !before.getIsInaccessible();
        boolean isAccessibleAfter = requestFailed ? isAccessibleBefore : (after != null);
        if (isAccessibleAfter != isAccessibleBefore) {
            changes.process(!isAccessibleAfter, ContentPromotionContent.IS_INACCESSIBLE);
        }

        String metadataAfter = Optional.ofNullable(after)
                .map(ContentPromotionContent::getMetadata)
                .map(t -> t.replaceAll("[^\\u0000-\\uFFFF]", "\uFFFD")) // cleanup 4-byte characters
                .orElse(null);
        BigInteger metadataHashAfter = Optional.ofNullable(metadataAfter)
                .map(this::calcContentPromotionJsonHash)
                .orElse(null);

        if (!requestFailed && isAccessibleAfter && !Objects.equals(before.getMetadataHash(), metadataHashAfter)) {
            changes.process(metadataHashAfter, ContentPromotionContent.METADATA_HASH);
            changes.process(metadataAfter, ContentPromotionContent.METADATA);
            changes.process(after.getPreviewUrl(), ContentPromotionContent.PREVIEW_URL);
        }

        if (changes.isAnyPropChanged()) {
            changes.process(updateTime, ContentPromotionContent.METADATA_MODIFY_TIME);
        }
        if (!requestFailed) {
            changes.process(updateTime, ContentPromotionContent.METADATA_REFRESH_TIME);
        }

        return changes.applyTo(before);
    }

    private BigInteger calcContentPromotionJsonHash(String contentJson) {
        return contentPromotionCoreTypeSupportFacade.calcMetadataHash(getContentType(), contentJson);
    }
}
