package ru.yandex.direct.jobs.contentpromotion.video;

import java.util.List;
import java.util.Optional;
import java.util.UUID;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Preconditions;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod;
import ru.yandex.direct.core.entity.banner.repository.BannerCommonRepository;
import ru.yandex.direct.core.entity.banner.type.contentpromo.BannerWithContentPromotionRepository;
import ru.yandex.direct.core.entity.contentpromotion.model.ContentPromotionContent;
import ru.yandex.direct.core.entity.contentpromotion.model.ContentPromotionContentType;
import ru.yandex.direct.core.entity.contentpromotion.repository.ContentPromotionRepository;
import ru.yandex.direct.core.entity.contentpromotion.type.ContentPromotionCoreTypeSupportFacade;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.jobs.contentpromotion.common.AbstractUpdateContentJob;
import ru.yandex.direct.jobs.contentpromotion.common.ContentPromotionJobConfiguration;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification;
import ru.yandex.direct.juggler.check.model.NotificationRecipient;
import ru.yandex.direct.libs.video.VideoClient;
import ru.yandex.direct.libs.video.model.VideoBanner;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.utils.JsonUtils;

import static java.util.Collections.singletonList;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_API_TEAM;

/**
 * Обновляет данные видео-контента в таблице ppc.content_promotion
 * Логика совпадает с UpdateVideoContentJob, просто обновляет данные из другой таблицы
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(seconds = 60 * 20 * 3 + 60 * 5),
        needCheck = ProductionOnly.class,
        tags = {DIRECT_API_TEAM},
        notifications = @OnChangeNotification(
                recipient = NotificationRecipient.CHAT_API_CLEAN_MONITORING,
                method = NotificationMethod.TELEGRAM,
                status = {JugglerStatus.OK, JugglerStatus.CRIT}
        )
)
@Hourglass(periodInSeconds = 60 * 20)
@ParametersAreNonnullByDefault
public class UpdateVideoFromContentPromotionJob extends AbstractUpdateContentJob {

    private static final String PROTOCOL_PREFIX = "https:";
    private static final String CLIENT_LOGIN = "update-video-content-job";

    private static final Logger LOGGER = LoggerFactory.getLogger(UpdateVideoFromContentPromotionJob.class);

    private final VideoClient videoClient;

    @Autowired
    public UpdateVideoFromContentPromotionJob(
            BannerWithContentPromotionRepository bannerContentPromotionRepository,
            ContentPromotionRepository contentPromotionRepository,
            BannerCommonRepository bannerCommonRepository,
            ContentPromotionCoreTypeSupportFacade
                    contentPromotionCoreTypeSupportFacade,
            DslContextProvider dslContextProvider,
            VideoClient videoClient) {
        super(bannerContentPromotionRepository, contentPromotionRepository, bannerCommonRepository,
                contentPromotionCoreTypeSupportFacade,
                dslContextProvider);
        this.videoClient = videoClient;
    }

    public UpdateVideoFromContentPromotionJob(int shard,
                                              BannerWithContentPromotionRepository bannerContentPromotionRepository,
                                              ContentPromotionRepository contentPromotionRepository,
                                              BannerCommonRepository bannerCommonRepository,
                                              ContentPromotionCoreTypeSupportFacade
                                                      contentPromotionCoreTypeSupportFacade,
                                              DslContextProvider dslContextProvider,
                                              VideoClient videoClient) {
        super(shard, bannerContentPromotionRepository, contentPromotionRepository, bannerCommonRepository,
                contentPromotionCoreTypeSupportFacade,
                dslContextProvider);
        this.videoClient = videoClient;
    }

    @Override
    protected ContentPromotionJobConfiguration getJobConfiguration() {
        return new ContentPromotionVideoJobConfiguration();
    }

    @Override
    protected void log(String messageFormat, Object... arguments) {
        LOGGER.info(messageFormat, arguments);
    }

    @Override
    protected ContentPromotionContentType getContentType() {
        return ContentPromotionContentType.VIDEO;
    }

    @Override
    protected List<ContentPromotionContent> getNewValues(@Nonnull List<ContentPromotionContent> oldValues) {
        // url в том же порядке, что и исходное видео
        List<String> urls = StreamEx.of(oldValues).map(ContentPromotionContent::getUrl).toList();
        // Клиент возвращает видеобаннеры в том же порядке, в каком получает url
        List<VideoBanner> videoBanners = StreamEx.of(videoClient.getMeta(urls, requestIdSupplier().get(), CLIENT_LOGIN))
                .zipWith(StreamEx.of(urls), (vb, url) ->
                        Optional.ofNullable(vb)
                                // Пробуем сделать запрос еще раз
                                .orElseGet(() -> videoClient.getMeta(singletonList(url),
                                        requestIdSupplier().get(), CLIENT_LOGIN).get(0)))
                .toList();
        List<ContentPromotionContent> newValues = videoBanners.stream()
                .map(UpdateVideoFromContentPromotionJob::convertToContentPromotionContent)
                .collect(Collectors.toList());
        StreamEx.zip(oldValues, newValues, (oldV, newV) -> {
            if (newV != null) {
                Preconditions.checkState(oldV.getUrl().endsWith(newV.getUrl()),
                        "Expected ContentPromotionContent video with url matching {}, but got with url {}",
                        oldV.getUrl(), newV.getUrl());
            }
            return null;
        });
        return newValues;
    }

    private static Supplier<String> requestIdSupplier() {
        return () -> UUID.randomUUID().toString().toUpperCase();
    }

    private static ContentPromotionContent convertToContentPromotionContent(@Nullable VideoBanner videoBanner) {
        if (videoBanner == null) {
            return null;
        }
        return new ContentPromotionContent()
                .withMetadata(JsonUtils.toJson(videoBanner))
                .withPreviewUrl(PROTOCOL_PREFIX + videoBanner.getThmbHref());
    }
}
