package ru.yandex.direct.jobs.conversionsource

import org.springframework.beans.factory.annotation.Autowired
import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod
import ru.yandex.direct.core.entity.conversionsource.model.ConversionActionName
import ru.yandex.direct.core.entity.conversionsource.model.ConversionSource
import ru.yandex.direct.core.entity.conversionsource.model.ConversionSourceId
import ru.yandex.direct.core.entity.conversionsource.service.ConversionSourceService
import ru.yandex.direct.core.entity.conversionsource.validation.ACTION_NAME_IN_PROGRESS
import ru.yandex.direct.core.entity.conversionsource.validation.ACTION_NAME_PAID
import ru.yandex.direct.env.NonDevelopmentEnvironment
import ru.yandex.direct.juggler.JugglerStatus
import ru.yandex.direct.juggler.check.annotation.JugglerCheck
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification
import ru.yandex.direct.juggler.check.model.CheckTag
import ru.yandex.direct.juggler.check.model.NotificationRecipient
import ru.yandex.direct.metrika.client.MetrikaClient
import ru.yandex.direct.metrika.client.model.response.CounterGoal
import ru.yandex.direct.scheduler.Hourglass
import ru.yandex.direct.scheduler.support.DirectJob

// Сколько источников конверсии обрабатывать за раз
private const val SYNC_CHUNK_SIZE = 1000

private val CDP_ORDER_TYPES = mapOf(
    CounterGoal.Type.CDP_ORDER_PAID to ACTION_NAME_PAID,
    CounterGoal.Type.CDP_ORDER_IN_PROGRESS to ACTION_NAME_IN_PROGRESS
)

/**
 * По источникам конверсий, в которых не указаны цели в конверсионных действиях, получает из метрики цели
 * для соответствующего счётчика и записывает подходящую цель в конверсионное действие.
 *
 * Этот процесс нужен, т.к. на момент создания источника конверсии, цели в метрике (в которые загружается через CRM API)
 * могут быть не созданы. Цели создаются после загрузки конверсий в CRM API, создаются с определёнными типами,
 * указанными в `CDP_ORDER_TYPES`. По этим типам мы их определяем и подгружаем в Директ.
 */
@JugglerCheck(
    ttl = JugglerCheck.Duration(minutes = 150),
    needCheck = NonDevelopmentEnvironment::class,
    tags = [CheckTag.DIRECT_PRIORITY_2],
    notifications = [OnChangeNotification(
        recipient = [NotificationRecipient.CHAT_CONVERSION_CENTER_BACK],
        method = [NotificationMethod.TELEGRAM],
        status = [JugglerStatus.OK, JugglerStatus.CRIT]
    )]
)
@Hourglass(periodInSeconds = 60 * 60, needSchedule = NonDevelopmentEnvironment::class)
class SyncGoalsFromMetrikaJob(
    private val metrikaClient: MetrikaClient,
    private val conversionSourceService: ConversionSourceService,
    private val chunkSize: Int = SYNC_CHUNK_SIZE,
) : DirectJob() {

    @Autowired
    constructor(metrikaClient: MetrikaClient, conversionSourceService: ConversionSourceService) :
        this(metrikaClient, conversionSourceService, SYNC_CHUNK_SIZE)

    override fun execute() {
        conversionSourceService.getSourcesWithoutGoals().chunked(chunkSize)
            .forEach { chunkOfSources ->
                val goalsToUpdate = getConversionActionGoals(chunkOfSources)
                conversionSourceService.setConversionActionGoals(goalsToUpdate)
            }
    }

    /**
     * Подготовить инструкции для обновления целей
     */
    private fun getConversionActionGoals(
        sources: List<ConversionSource>,
    ): Map<ConversionSourceId, Map<ConversionActionName, GoalId>> {
        val counterToSource = sources.associate { it.counterId.toInt() to it.id }
        val goalData = getCdpOrderGoals(counterToSource.keys)
        return goalData.asSequence()
            .map { (counterId, cdpOrderGoals) ->
                counterId.toLong() to cdpOrderGoals.mapKeys { (type, _) -> CDP_ORDER_TYPES[type]!! }
            }
            .toMap()
    }

    /**
     * Получить из метрики цели типов CDP_ORDER_* по указанным счётчикам
     */
    private fun getCdpOrderGoals(
        counterIds: Set<MetrikaCounterId>,
    ): Map<MetrikaCounterId, Map<CounterGoal.Type, GoalId>> {
        return metrikaClient.getMassCountersGoalsFromMetrika(counterIds).mapValues { (_, goalInfo) ->
            goalInfo.filter { it.type in CDP_ORDER_TYPES }.associate { it.type!! to it.id.toLong() }
        }
    }
}

private typealias MetrikaCounterId = Int
private typealias GoalId = Long
