package ru.yandex.direct.jobs.copy;

import java.time.Duration;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod;
import ru.yandex.direct.core.copyentity.CopyResult;
import ru.yandex.direct.core.copyentity.model.CampaignCopyJobParams;
import ru.yandex.direct.core.copyentity.model.CampaignCopyJobResult;
import ru.yandex.direct.core.copyentity.model.CopyCampaignFlags;
import ru.yandex.direct.core.entity.campaign.service.CopyCampaignService;
import ru.yandex.direct.dbqueue.JobFailedPermanentlyException;
import ru.yandex.direct.dbqueue.JobFailedWithTryLaterException;
import ru.yandex.direct.dbqueue.model.DbQueueJob;
import ru.yandex.direct.dbqueue.service.DbQueueService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.env.NonDevelopmentEnvironment;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.jobs.copy.copyreport.CopyReportContext;
import ru.yandex.direct.jobs.copy.copyreport.CopyReportService;
import ru.yandex.direct.jobs.copy.copyreport.CopyReportType;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification;
import ru.yandex.direct.juggler.check.model.NotificationRecipient;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectShardedJob;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceHelper;
import ru.yandex.direct.tracing.real.RealTrace;
import ru.yandex.direct.tracing.util.TraceUtil;
import ru.yandex.direct.utils.ThreadUtils;

import static ru.yandex.direct.core.entity.dbqueue.DbQueueJobTypes.CAMPAIGNS_COPY;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_1_NOT_READY;

/**
 * Офлайн-копирование кампаний через dbqueue
 * для локального запуска нужно передавать --shard [номер шарда]
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(hours = 1, minutes = 30),
        needCheck = ProductionOnly.class,
        notifications = @OnChangeNotification(
                recipient = NotificationRecipient.CHAT_CAMPAIGN_COPY_MONITORING,
                method = NotificationMethod.TELEGRAM,
                status = {JugglerStatus.OK, JugglerStatus.CRIT}
        ),
        tags = {DIRECT_PRIORITY_1_NOT_READY})
@Hourglass(periodInSeconds = 5, needSchedule = NonDevelopmentEnvironment.class)
public class CampaignCopyJob extends DirectShardedJob {
    private static final Logger logger = LoggerFactory.getLogger(CampaignCopyJob.class);

    private final DbQueueService dbQueueService;
    private final CopyCampaignService copyCampaignService;
    private final CopyReportService copyReportService;
    private final TraceHelper traceHelper;

    private static final int MAX_ATTEMPTS = 1;

    private static final long MAX_OPERATION_TIME_MS = 250_000L;
    private static final long SLEEP_DURATION_MS = 3_000L;

    private static final Duration TRY_LATER_DURATION = Duration.ofMinutes(5);

    @Autowired
    public CampaignCopyJob(
            DbQueueService dbQueueService,
            CopyCampaignService copyCampaignService,
            CopyReportService copyReportService,
            TraceHelper traceHelper) {
        this.dbQueueService = dbQueueService;
        this.copyCampaignService = copyCampaignService;
        this.copyReportService = copyReportService;
        this.traceHelper = traceHelper;
    }

    @Override
    public void execute() {
        long startTime = System.currentTimeMillis();
        while (System.currentTimeMillis() - startTime < MAX_OPERATION_TIME_MS) {
            if (!dbQueueService.grabAndProcessJob(getShard(), CAMPAIGNS_COPY,
                    this::processGrabbedJobWrapper, MAX_ATTEMPTS, this::onError)) {
                ThreadUtils.sleep(SLEEP_DURATION_MS);
            }
        }
    }

    private CampaignCopyJobResult processGrabbedJobWrapper(
            DbQueueJob<CampaignCopyJobParams, CampaignCopyJobResult> jobInfo) {
        // Если предыдущее выполнение задачи упало не записав результат, задача может быть выбрана еще раз
        // Явно запрещаем такой задаче выполниться повторно
        if (jobInfo.getTryCount() > MAX_ATTEMPTS) {
            throw new JobFailedPermanentlyException("job was already grabbed");
        }

        try (var ignored = traceHelper.guard(buildJobTrace())) {
            return processGrabbedJob(jobInfo);
        } catch (Exception e) {
            // умеем выбрасывать checked исключения из kotlin кода
            ThreadUtils.checkInterrupted(e);

            throw new JobFailedWithTryLaterException(TRY_LATER_DURATION, "copy job failed", e);
        }
    }

    private CampaignCopyJobResult processGrabbedJob(
            DbQueueJob<CampaignCopyJobParams, CampaignCopyJobResult> jobInfo) {
        CampaignCopyJobParams params = jobInfo.getArgs();

        long campaignId = params.getCampaignId();
        ClientId clientIdFrom = params.getClientIdFrom();
        ClientId clientIdTo = params.getClientIdTo();
        long operatorUid = jobInfo.getUid();

        CopyCampaignFlags flags = params.getFlags();

        logger.info("Grabbed job {}: will copy campaigns {} from client {} to client {} by operator {} with flags {}",
                jobInfo.getId(), campaignId, clientIdFrom, clientIdTo, operatorUid, flags);

        CopyResult<Long> copyResult =
                copyCampaignService.copyCampaigns(clientIdFrom, clientIdTo, operatorUid, List.of(campaignId), flags);

        List<Result<Long>> campaignResults = copyResult.getMassResult().toResultList();
        boolean failed = !campaignResults.isEmpty() && campaignResults.get(0).isSuccessful();

        if (failed) {
            logger.info("Successfully copied campaign {}", campaignId);
        } else {
            logger.info("Failed to copy campaign {}", campaignId);
        }

        if (flags.isGenerateReport()) {
            Map<CopyReportType, String> reports = copyReportService.saveReport(new CopyReportContext(
                    jobInfo.getId(),
                    jobInfo.getClientId().asLong(),
                    jobInfo.getArgs().getLoginFrom(),
                    jobInfo.getArgs().getLoginTo(),
                    copyResult.getEntityContext(),
                    failed ? Set.of(campaignId) : Set.of()
            ));
            return new CampaignCopyJobResult(
                    null,
                    reports.getOrDefault(CopyReportType.BANNERS, null),
                    reports.getOrDefault(CopyReportType.PHRASES, null),
                    reports.getOrDefault(CopyReportType.RETARGETINGS, null),
                    reports.getOrDefault(CopyReportType.DYNAMIC_CONDITIONS, null),
                    reports.getOrDefault(CopyReportType.PERFORMANCE_FILTERS, null)
            );
        }

        return new CampaignCopyJobResult(null);
    }

    private CampaignCopyJobResult onError(
            DbQueueJob<CampaignCopyJobParams, CampaignCopyJobResult> jobInfo, String stackTrace) {
        return new CampaignCopyJobResult(stackTrace);
    }

    /**
     * Каждая задача копирования выполняется в отдельном trace, таким образом по span_id можно посмотреть лог отдельной
     * кампании
     */
    private Trace buildJobTrace() {
        return RealTrace.builder()
                .withIds(Trace.current().getTraceId(), Trace.current().getSpanId(), TraceUtil.randomId())
                .withService(Trace.current().getService())
                .withMethod(Trace.current().getMethod())
                .withTags(Trace.current().getTags())
                .build();
    }
}
