package ru.yandex.direct.jobs.copy.copyreport;

import java.io.ByteArrayOutputStream;
import java.io.OutputStreamWriter;
import java.io.Writer;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import com.univocity.parsers.csv.CsvWriter;
import com.univocity.parsers.csv.CsvWriterSettings;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.banner.model.BannerWithAdGroupId;
import ru.yandex.direct.core.entity.keyword.model.Keyword;
import ru.yandex.direct.core.entity.mdsfile.model.MdsFileSaveRequest;
import ru.yandex.direct.core.entity.mdsfile.service.MdsFileService;
import ru.yandex.direct.core.entity.retargeting.model.Retargeting;
import ru.yandex.direct.model.Entity;

import static ru.yandex.direct.core.entity.mdsfile.model.MdsStorageType.CAMP_COPY_REPORT;

@Service
public class CopyReportService {
    private static final String CAMPAIGN_ID = "CampaignId_";
    private static final String ADGROUP_ID = "AdGroupId_";
    private static final String BANNER_ID = "BannerId_";
    private static final String KEYWORD_ID = "KeywordId_";
    private static final String RETARGETING_CONDITION_ID = "RetargetingConditionId_";
    private final CsvWriterSettings csvWriterSettings = new CsvWriterSettings();
    private final MdsFileService mdsFileService;
    private CopyReportContext context;

    @Autowired
    public CopyReportService(MdsFileService mdsFileService) {
        this.mdsFileService = mdsFileService;
    }

    public Map<CopyReportType, String> saveReport(CopyReportContext context) {
        Objects.requireNonNull(context.loginFrom, "loginFrom must not be null");
        Objects.requireNonNull(context.loginTo, "loginTo must not be null");
        this.context = context;
        saveBannersReport();
        savePhrasesReport();
        saveRetargetingsReport();

        if (!context.requests.isEmpty()) {
            mdsFileService.saveMdsFiles(context.requests, context.clientId);
        }
        return context.result;
    }

    private void saveBannersReport() {
        saveTypicalReport(BannerWithAdGroupId.class, CopyReportType.BANNERS, BANNER_ID);
    }

    private void savePhrasesReport() {
        saveTypicalReport(Keyword.class, CopyReportType.PHRASES, KEYWORD_ID);
    }

    private void saveRetargetingsReport() {
        saveTypicalReport(Retargeting.class, CopyReportType.RETARGETINGS, RETARGETING_CONDITION_ID);
    }

    private void saveTypicalReport(Class<? extends Entity<Long>> thirdClass,
                                   CopyReportType copyReportType,
                                   String thirdIdHeader) {
        Map<Long, Long> thirdIdsMap = context.entityContext.getCopyMappingByClass(thirdClass);
        if (!context.entityContext.getEntityClasses().contains(thirdClass) || thirdIdsMap.isEmpty()) {
            return;
        }
        Map<Long, Set<Long>> secondOldIdsToThirdOldIds = context.entityContext.getIdRelations(
                AdGroup.class,
                thirdClass);

        List<String[]> rows = new ArrayList<>();
        boolean hasData = generateRows(rows,
                secondOldIdsToThirdOldIds,
                thirdIdsMap);
        if (hasData) {
            String[] headers = {
                    CAMPAIGN_ID + context.loginFrom,
                    ADGROUP_ID + context.loginFrom,
                    thirdIdHeader + context.loginFrom,
                    CAMPAIGN_ID + context.loginTo,
                    ADGROUP_ID + context.loginTo,
                    thirdIdHeader + context.loginTo,
            };
            String name = generateName(copyReportType);
            saveFileDataRequest(generateFileData(headers, rows), name);
            context.result.put(copyReportType, name);
        }
    }

    private boolean generateRows(List<String[]> rows,
                                 Map<Long, Set<Long>> secondOldIdsToThirdOldIds,
                                 Map<Long, Long> thirdIdsMap) {
        if (secondOldIdsToThirdOldIds.isEmpty() || thirdIdsMap.isEmpty()) {
            return false;
        }
        boolean hasData = false;
        for (Map.Entry<Long, Set<Long>> campaignEntry : context.oldCampaignIdsToOldAdGroupIds.entrySet()) {
            Long oldCampaignId = campaignEntry.getKey();
            if (context.failedCampaignIds.contains(oldCampaignId)) {
                continue;
            }
            for (Long oldAdGroupId : campaignEntry.getValue()) {
                for (Long thirdOldId : secondOldIdsToThirdOldIds.getOrDefault(oldAdGroupId, Set.of())) {
                    rows.add(new String[]{
                            oldCampaignId.toString(),
                            oldAdGroupId.toString(),
                            thirdOldId.toString(),
                            toStringWithErrors(context.oldCampaignIdsToNewCampaignIds.get(oldCampaignId)),
                            toStringWithErrors(context.oldAdGroupIdsToNewAdGroupIds.get(oldAdGroupId)),
                            toStringWithErrors(thirdIdsMap.get(thirdOldId)),
                    });
                    if (thirdIdsMap.get(thirdOldId) != null) {
                        hasData = true;
                    }
                }
            }
        }
        return hasData;
    }

    private String toStringWithErrors(Long id) {
        return id == null ? "Error" : id.toString();
    }

    private String generateName(CopyReportType type) {
        return String.format("%s_from_%s_to_%s_%s",
                type.name().toLowerCase(),
                context.loginFrom,
                context.loginTo,
                context.jobId);
    }

    private ByteArrayOutputStream generateFileData(String[] headers, List<String[]> rows) {
        ByteArrayOutputStream csvResult = new ByteArrayOutputStream();
        Writer outputWriter = new OutputStreamWriter(csvResult);
        CsvWriter writer = new CsvWriter(outputWriter, csvWriterSettings);
        writer.writeHeaders(headers);
        writer.writeRowsAndClose(rows.toArray(new String[0][]));
        writer.close();
        return csvResult;
    }

    private void saveFileDataRequest(ByteArrayOutputStream filedata, String name) {
        MdsFileSaveRequest request = new MdsFileSaveRequest(CAMP_COPY_REPORT, filedata.toByteArray())
                .withCustomName(name);
        context.requests.add(request);
    }
}
