package ru.yandex.direct.jobs.cpaautobudget;

import java.time.LocalDateTime;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.jooq.DSLContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod;
import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.core.entity.autobudget.model.ConversionLevelToUpdate;
import ru.yandex.direct.core.entity.autobudget.model.GoalIdWithConversionLevel;
import ru.yandex.direct.core.entity.autobudget.repository.GoalQualityYtRepository;
import ru.yandex.direct.dbschema.ppcdict.Tables;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.env.NonProductionEnvironment;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapper.JooqModelToDbFieldValuesMapper;
import ru.yandex.direct.jooqmapperhelper.UpdateHelper;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification;
import ru.yandex.direct.juggler.check.model.NotificationRecipient;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectJob;

import static ru.yandex.direct.common.db.PpcPropertyNames.ATTEMPT_NUMBER_FOR_FULL_UPDATE_CONVERSION_LEVEL;
import static ru.yandex.direct.common.db.PpcPropertyNames.CONVERSION_LEVEL_UPDATE_LIMIT;
import static ru.yandex.direct.common.db.PpcPropertyNames.GOAL_QUALITY_CONVERSION_LEVEL_LAST_UPDATE_TIME;
import static ru.yandex.direct.dbschema.ppcdict.tables.MetrikaGoals.METRIKA_GOALS;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_2;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_SPB_SERVER_SIDE_TEAM;

/**
 * Обновляет conversion_level в таблице metrika_goals.
 * Источником данных является YT таблица экспортируемая из ОКР:
 * {@link ru.yandex.direct.core.entity.autobudget.ytmodels.generated.YtDbTables#GOALQUALITY}, где значения
 * conversion_level представлены тремя колонками ({@code IsPerfectGoal, IsGoodGoal, IsNotBadGoal}
 * https://st.yandex-team.ru/DIRECT-137142)
 * Сохраняем в ppcdict в таблицу: {@link Tables#METRIKA_GOALS}.
 * <p>
 * Джоба сравнивает синхронизированные с прода значения metrika_goals.conversion_level
 * (См. https://yt.yandex-team.ru/hahn/navigation?offsetMode=key&path=//home/direct/mysql-sync/v.16/ppcdict/straight/metrika_goals)
 * со значениями в таблице GoalQuality, получает расхождения и далее обновляет записи в {@link Tables#METRIKA_GOALS}.
 * Количество таких данных для обновления не должно превышать {@link UpdateConversionLevelForMetrikaGoalsJob#DEFAULT_CONVERSION_LEVEL_UPDATE_LIMIT},
 * в противном случае обновление не будет выполнено для всех записей, что будет отражено в логах.
 * Выполнение джобы на проде в большенстве случае должно происходить после ее запуска на тестовых средах,
 * так как таблица в yt, используемая для сравнения текущих значений conversion_level со значениями из ОКР (GoalQuality)
 * синхронизируется с продовыми, и если на тестах джоба запустится позже - там значения для обновления уже не будут обнаружены.
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(days = 2),
        needCheck = ProductionOnly.class,
        tags = {DIRECT_PRIORITY_2, DIRECT_SPB_SERVER_SIDE_TEAM},
        notifications = @OnChangeNotification(
                recipient = NotificationRecipient.LOGIN_DMITANOSH,
                method = NotificationMethod.TELEGRAM,
                status = {JugglerStatus.OK, JugglerStatus.CRIT}
        )
)
@Hourglass(cronExpression = "0 0 10,22 * * ?", needSchedule = ProductionOnly.class)
@JugglerCheck(ttl = @JugglerCheck.Duration(days = 2),
        needCheck = NonProductionEnvironment.class,
        tags = {DIRECT_PRIORITY_2, DIRECT_SPB_SERVER_SIDE_TEAM}
)
@Hourglass(cronExpression = "0 0 7,19 * * ?", needSchedule = NonProductionEnvironment.class)
@ParametersAreNonnullByDefault
public class UpdateConversionLevelForMetrikaGoalsJob extends DirectJob {

    private static final Logger logger = LoggerFactory.getLogger(UpdateConversionLevelForMetrikaGoalsJob.class);

    private static final long DEFAULT_CONVERSION_LEVEL_UPDATE_LIMIT = 100_000;
    private static final int UPDATE_CHUNK_SIZE = 1000;
    private static final int MAX_ATTEMPTS_FOR_FULL_UPDATE = 3;

    private static final JooqModelToDbFieldValuesMapper<GoalIdWithConversionLevel> MAPPER = getMapper();

    private final GoalQualityYtRepository goalQualityYtRepository;
    private final PpcPropertiesSupport ppcPropertiesSupport;
    private final DslContextProvider dslContextProvider;

    @Autowired
    public UpdateConversionLevelForMetrikaGoalsJob(GoalQualityYtRepository goalQualityYtRepository,
                                                   PpcPropertiesSupport ppcPropertiesSupport,
                                                   DslContextProvider dslContextProvider) {
        this.goalQualityYtRepository = goalQualityYtRepository;
        this.ppcPropertiesSupport = ppcPropertiesSupport;
        this.dslContextProvider = dslContextProvider;
    }

    @Override
    public void execute() {
        PpcProperty<LocalDateTime> lastModificationTimeProperty =
                ppcPropertiesSupport.get(GOAL_QUALITY_CONVERSION_LEVEL_LAST_UPDATE_TIME);
        LocalDateTime lastTableModificationTime = goalQualityYtRepository.getTableLastUpdateTime();

        if (!needUpdate(lastTableModificationTime, lastModificationTimeProperty)) {
            logger.info("Table has already been updated before. Skipping.");
            return;
        }

        PpcProperty<Integer> attemptNumberForFullUpdateProperty =
                ppcPropertiesSupport.get(ATTEMPT_NUMBER_FOR_FULL_UPDATE_CONVERSION_LEVEL);
        Integer currentAttemptNumberForFullUpdate = attemptNumberForFullUpdateProperty.getOrDefault(0);
        if (currentAttemptNumberForFullUpdate != 0) {
            logger.info("Current attempt for full update conversion level is {}", currentAttemptNumberForFullUpdate);
        }

        long updateLimit = ppcPropertiesSupport.get(CONVERSION_LEVEL_UPDATE_LIMIT)
                .getOrDefault(DEFAULT_CONVERSION_LEVEL_UPDATE_LIMIT);

        // из yt берутся не более указанного количества записей для обновления
        List<GoalIdWithConversionLevel> goalsWithConversionLevel =
                goalQualityYtRepository.getGoalsConversionLevelToUpdate(updateLimit);
        logger.info("Fetched {} changes of conversion_level from yt-table", goalsWithConversionLevel.size());

        boolean isFullyUpdated = goalsWithConversionLevel.size() < updateLimit;

        if (goalsWithConversionLevel.size() > 0) {
            int updated = updateConversionLevel(goalsWithConversionLevel);
            logger.info("Updated {} records of metrika_goals", updated);
        }

        if (isFullyUpdated) {
            logger.info("Setting {} property value to '{}'", GOAL_QUALITY_CONVERSION_LEVEL_LAST_UPDATE_TIME.getName(),
                    lastTableModificationTime);
            lastModificationTimeProperty.set(lastTableModificationTime);

            if (currentAttemptNumberForFullUpdate > 0) {
                setAttemptNumberForFullUpdate(attemptNumberForFullUpdateProperty, 0);
            }
        } else {
            currentAttemptNumberForFullUpdate += 1;
            setAttemptNumberForFullUpdate(attemptNumberForFullUpdateProperty, currentAttemptNumberForFullUpdate);

            // Если не удалось выполнить полное обновление более указанного количества раз - выходим с ошибкой
            // В таком случае нужно увеличить количество получаемых из yt таблицы записей
            if (currentAttemptNumberForFullUpdate > MAX_ATTEMPTS_FOR_FULL_UPDATE) {
                logger.warn("Number of attempts to fully update conversion_level exceeded the specified limit {}",
                        MAX_ATTEMPTS_FOR_FULL_UPDATE);
                setJugglerStatus(JugglerStatus.WARN,
                        "metrika_goals.conversion_level not fully updated from yt-table, check log for details");
            }

            logger.warn("Number of records to update exceeds {}. Non-updated records will be processed on the next run",
                    updateLimit);
        }
    }

    private int updateConversionLevel(List<GoalIdWithConversionLevel> goalsToUpdate) {
        List<AppliedChanges<GoalIdWithConversionLevel>> changes = StreamEx.of(goalsToUpdate)
                .map(goalToUpdate -> new ModelChanges<>(goalToUpdate.getId(), GoalIdWithConversionLevel.class)
                        .process(goalToUpdate.getConversionLevelToUpdate(),
                                GoalIdWithConversionLevel.CONVERSION_LEVEL_TO_UPDATE)
                        .applyTo(new GoalIdWithConversionLevel().withId(goalToUpdate.getId()))
                )
                .toList();

        return updateConversionLevel(dslContextProvider.ppcdict(), changes);
    }


    private int updateConversionLevel(DSLContext context, List<AppliedChanges<GoalIdWithConversionLevel>> changes) {
        return StreamEx.ofSubLists(changes, UPDATE_CHUNK_SIZE)
                .map(chunk -> new UpdateHelper<>(context, METRIKA_GOALS.GOAL_ID)
                        .processUpdateAll(MAPPER, chunk)
                        .execute())
                .foldLeft(0, Integer::sum);
    }

    private static boolean needUpdate(LocalDateTime lastTableModificationTime,
                                      PpcProperty<LocalDateTime> lastModificationTimeProperty) {
        logger.info("{} property value is '{}'. Yt-table modification time is {}",
                GOAL_QUALITY_CONVERSION_LEVEL_LAST_UPDATE_TIME.getName(),
                lastModificationTimeProperty.get(), lastTableModificationTime);
        return lastTableModificationTime.isAfter(lastModificationTimeProperty.getOrDefault(LocalDateTime.MIN));
    }

    private static void setAttemptNumberForFullUpdate(PpcProperty<Integer> attemptNumberForFullUpdateProperty,
                                                      Integer newAttemptNumberForFullUpdate) {
        logger.info("Setting {} property value to '{}'", ATTEMPT_NUMBER_FOR_FULL_UPDATE_CONVERSION_LEVEL.getName(),
                newAttemptNumberForFullUpdate);
        attemptNumberForFullUpdateProperty.set(newAttemptNumberForFullUpdate);
    }

    private static JooqModelToDbFieldValuesMapper<GoalIdWithConversionLevel> getMapper() {
        return JooqMapperWithSupplierBuilder.builder(GoalIdWithConversionLevel::new)
                .map(property(GoalIdWithConversionLevel.ID, METRIKA_GOALS.GOAL_ID))
                .map(convertibleProperty(
                        GoalIdWithConversionLevel.CONVERSION_LEVEL_TO_UPDATE, METRIKA_GOALS.CONVERSION_LEVEL,
                        ConversionLevelToUpdate::fromSource,
                        ConversionLevelToUpdate::toSource))
                .build();
    }
}
